/*
 * Copyright 2018 Google LLC
 *
 * Licensed for use under "ARCore Additional Terms of Service". You may obtain
 * a copy of the license at https://developers.google.com/ar/develop/terms.
 */

package com.google.ar.core;


/**
 * The SharedCamera class allows the app to share control of the camera with ARCore and obtain image
 * readers and surfaces for rendering. Images can be read concurrently while ARCore is running or
 * while ARCore is paused.
 *
 * <p>Use this class to obtain wrappers for the <a
 * href="https://developer.android.com/reference/android/hardware/camera2/CameraDevice">{@code
 * CameraDevice}</a> and <a
 * href="https://developer.android.com/reference/android/hardware/camera2/CameraCaptureSession.StateCallback">CameraCaptureSession
 * state callbacks</a>, so that camera events and data objects can be shared with ARCore.
 *
 * <p>When a session is created for shared camera access, ARCore's behavior changes in the following
 * ways:
 *
 * <ul>
 *   <li>While ARCore is active, the app can use Camera2 APIs, except it must not call <a
 *       href="https://developer.android.com/reference/android/hardware/camera2/CameraCaptureSession#setRepeatingRequest(android.hardware.camera2.CaptureRequest,%20android.hardware.camera2.CameraCaptureSession.CaptureCallback,%20android.os.Handler)">{@code
 *       CameraCaptureSession#setRepeatingRequest}</a> as this will interfere with ARCore operation.
 *       The Camera2 APIs can be used directly without restriction while ARCore is paused.
 *   <li>{@link com.google.ar.core.Session#getSharedCamera() Session#getSharedCamera()} can be used
 *       to obtain an instance of the {@link com.google.ar.core.SharedCamera SharedCamera} object.
 *   <li>Although ARCore will continue to track normally, it will not be able to use the depth
 *       sensor even if one is present on the device.
 * </ul>
 *
 * <p>Example of app activity using shared camera:
 *
 * <pre>{@code Session sharedSession;
 * SharedCamera sharedCamera;
 *
 * void onCreate(Bundle savedInstanceState) {
 *   sharedSession = new Session(context, EnumSet.of(Session.Feature.SHARED_CAMERA));
 *   openCameraForSharing();
 *   createCameraCaptureSession();
 *   // ...
 * }
 *
 * private void openCameraForSharing() {
 *
 *   sharedCamera = sharedSession.getSharedCamera();
 *
 *   // Use callback wrapper.
 *   CameraManager.openCamera(
 *       cameraID,
 *       sharedCamera.createARDeviceStateCallback(appDeviceStateCallback, appHandler),
 *       appHandler);
 * }
 *
 * private void createCameraCaptureSession() {
 *   // Get list of ARCore created surfaces. Required for ARCore tracking.
 *   ArrayList<Surface> surfaceList = sharedCamera.getSurfaces();
 *
 *   // (Optional) Add a custom CPU image reader surface on devices that support CPU image access.
 *   ImageReader cpuImageReader = ImageReader.newInstance(…);
 *   surfaceList.add(cpuImageReader.getSurface());
 *
 *   // Use callback wrapper.
 *   cameraDevice.createCaptureSession(
 *       surfaceList,
 *       sharedCamera.createARSessionStateCallback(
 *           appSessionStateCallback, appHandler),
 *       appHandler);
 * }}</pre>
 */
@SuppressWarnings({"unchecked", "deprecation", "all"})
public class SharedCamera {

  SharedCamera() {
    // Method contents removed.
  }

  /**
   * Set app created surfaces, to receive additional images when ARCore is active.
   *
   * <p>It is the app's responsibility to ensure that the {@link
   * android.hardware.camera2.CameraDevice} and ARCore both support the full set of
   * streams/surfaces/resolutions, those automatically created by ARCore and returned by {@link
   * #getArCoreSurfaces()}, and the app created surfaces specified here.
   *
   * <p>A single additional surface, with a resolution equal to one of the CPU resolutions returned
   * by {@link
   * com.google.ar.core.Session#getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter)
   * Session#getSupportedCameraConfigs(CameraConfigFilter)}, is guaranteed to be supported by the
   * {@link android.hardware.camera2.CameraDevice} and ARCore. Additional surfaces beyond one, and
   * resolutions not listed in {@link
   * com.google.ar.core.Session#getSupportedCameraConfigs(com.google.ar.core.CameraConfigFilter) },
   * might or might not be supported by the device and/or ARCore. It is the app developer's
   * responsibility to verify that ARCore and all targeted devices support the requested app
   * surfaces.
   *
   * <p>The provided list must only include any surfaces that are in addition to the ones returned
   * by {@link #getArCoreSurfaces()}.
   *
   * <p>Note, when calling {@link
   * android.hardware.camera2.CameraCaptureSession#setRepeatingRequest(android.hardware.camera2.CaptureRequest,
   * android.hardware.camera2.CameraCaptureSession.CaptureCallback, android.os.Handler)}, you must
   * pass in a complete list of surfaces that includes:
   *
   * <ul>
   *   <li>All ARCore surfaces returned by {@link #getArCoreSurfaces()}.
   *   <li>Any additional surface(s) you create and pass into {@link
   *       android.hardware.camera2.CameraDevice#createCaptureSession(android.hardware.camera2.params.SessionConfiguration)}
   *       and this method.
   * </ul>
   *
   * @param cameraId the ID of this camera device.
   * @param surfaces the list of a {@code Surface}.
   */
  public void setAppSurfaces(
      java.lang.String cameraId, java.util.List<android.view.Surface> surfaces) {
    // Method contents removed.
  }

  /**
   * Gets access to the SurfaceTexture provided by ARCore.
   *
   * <p>This method should be called after the camera has been successfully opened, for example in
   * the {@code onOpened} callback in {@code android.hardware.camera2.CameraDevice.StateCallback}.
   *
   * <p>ARCore will not attach the SurfaceTexture directly to the textures from {@link
   * com.google.ar.core.Session#setCameraTextureName(int) } or {@link
   * com.google.ar.core.Session#setCameraTextureNames(int[]) Session#setCameraTextureNames(int[])}
   * if <a href="https://developers.google.com/ar/develop/java/buffer-multiple-frames">frame
   * buffering</a> is enabled. The app must detach and reattach to the desired texture in order to
   * continue using this object while ARCore is paused.
   */
  public android.graphics.SurfaceTexture getSurfaceTexture() {
    // Method contents removed.
  }

  /**
   * Gets access to the list of surfaces created and used by ARCore.
   *
   * <p>Multiple calls to this function will return the same list.
   *
   * <p>ARCore always creates two surfaces. A third surface is optionally created if requested by
   * the app by selecting a non-default camera config. The order of surfaces in the returned list is
   * guaranteed to be:
   *
   * <ol>
   *   <li>GPU texture surface (best available resolution, typically 1920x1280)
   *   <li>Motion tracking surface (640x480 resolution)
   *   <li>(Optional) Higher resolution CPU image, if set by the app using {@link
   *       com.google.ar.core.Session#setCameraConfig(com.google.ar.core.CameraConfig) }
   * </ol>
   *
   * <p>Note, when calling {@link android.hardware.camera2.CameraCaptureSession#setRepeatingRequest(
   * android.hardware.camera2.CaptureRequest,
   * android.hardware.camera2.CameraCaptureSession.CaptureCallback,android.os.Handler)}, you must
   * pass in a complete list of surfaces that includes:
   *
   * <ul>
   *   <li>All ARCore surfaces returned by this method.
   *   <li>Any additional surface(s) you create and passed into {@link
   *       android.hardware.camera2.CameraDevice#createCaptureSession(
   *       android.hardware.camera2.params.SessionConfiguration)} and {@link
   *       #setAppSurfaces(java.lang.String,java.util.List)}.
   * </ul>
   *
   * @return list of the surfaces created by ARCore. Does not include additional surfaces created by
   *     the app.
   */
  public java.util.List<android.view.Surface> getArCoreSurfaces() {
    // Method contents removed.
  }

  /**
   * The openCamera command on the CameraManager should use this wrapped callback as input.
   *
   * <pre>CameraManager.openCamera(
   *   cameraID,
   *   sharedCamera.createARDeviceStateCallback(appDeviceStateCallback, appHandler),
   *   appHandler);
   * }</pre>
   *
   * See {@link com.google.ar.core.SharedCamera SharedCamera} class level documentation for a more
   * complete example.
   */
  public android.hardware.camera2.CameraDevice.StateCallback createARDeviceStateCallback(
      android.hardware.camera2.CameraDevice.StateCallback appCallback,
      android.os.Handler appHandler) {
    // Method contents removed.
  }

  /**
   * Wraps a {@link android.hardware.camera2.CameraCaptureSession.StateCallback
   * CameraCaptureSession.StateCallback} to allow ARCore to control the camera. The result of this
   * call should be used in <a
   * href="https://developer.android.com/reference/android/hardware/camera2/CameraDevice#createCaptureSession(java.util.List%3Candroid.view.Surface%3E,%20android.hardware.camera2.CameraCaptureSession.StateCallback,%20android.os.Handler)">{@code
   * CameraDevice#createCaptureSession}</a> as follows:
   *
   * <pre>{@code cameraDevice.createCaptureSession(
   *   surfaceList,
   *   sharedCamera.createARSessionStateCallback(appSessionStateCallback, appHandler),
   *   appHandler);
   * }</pre>
   *
   * See {@link com.google.ar.core.SharedCamera SharedCamera} class level documentation for a more
   * complete example.
   */
  public android.hardware.camera2.CameraCaptureSession.StateCallback createARSessionStateCallback(
      android.hardware.camera2.CameraCaptureSession.StateCallback appCallback,
      android.os.Handler appHandler) {
    // Method contents removed.
  }

  /**
   * Set {@link android.hardware.camera2.CameraCaptureSession.CaptureCallback
   * CameraCaptureSession.CaptureCallback} by sending it to ARCore in order to provide metadata
   * information to the client.
   */
  public void setCaptureCallback(
      android.hardware.camera2.CameraCaptureSession.CaptureCallback appCaptureCallback,
      android.os.Handler appHandler) {
    // Method contents removed.
  }
}
