package com.gradle.publish;

import org.apache.maven.model.Dependency;
import org.apache.maven.model.Exclusion;
import org.gradle.api.Project;
import org.gradle.api.artifacts.*;
import org.gradle.api.publish.PublicationContainer;
import org.gradle.api.publish.PublishingExtension;
import org.gradle.api.publish.maven.MavenPublication;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public class DependenciesBuilder {
    private static final String API_ELEMENTS = "apiElements";
    private static final String RUNTIME_ELEMENTS = "runtimeElements";

    private static final Logger LOGGER = LoggerFactory.getLogger(DependenciesBuilder.class);

    public List<Dependency> buildMavenDependencies(ConfigurationContainer configurations) {
        Configuration compileConfig = configurations.getByName(API_ELEMENTS);
        Configuration runtimeConfig = configurations.getByName(RUNTIME_ELEMENTS);
        return getDependencies(compileConfig, runtimeConfig);
    }

    private List<Dependency> getDependencies(Configuration compileConfig, Configuration runtimeConfig) {
        List<Dependency> mavenDependencies = new ArrayList<Dependency>();
        addUniqueScopedDependencies(mavenDependencies, compileConfig, "compile");
        addUniqueScopedDependencies(mavenDependencies, runtimeConfig, "runtime");
        return mavenDependencies;
    }

    private void addUniqueScopedDependencies(List<Dependency> mavenDependencies, Configuration config, String scope) {
        for (org.gradle.api.artifacts.Dependency dependency : config.getAllDependencies()) {
            org.gradle.api.artifacts.Dependency refinedDependency = (dependency instanceof ProjectDependency)
                    ? preparePublishedProjectDependency((ProjectDependency) dependency)
                    : dependency;
            if (refinedDependency instanceof ModuleDependency &&
                    !isDuplicateDependency(mavenDependencies, (ModuleDependency) refinedDependency)) {
                addDependency(mavenDependencies, (ModuleDependency) refinedDependency, scope);
            }
        }
    }

    private ModuleDependency preparePublishedProjectDependency(ProjectDependency projectDependency) {
        Project project = projectDependency.getDependencyProject();
        PublishingExtension publishingExtension = project.getExtensions().findByType(PublishingExtension.class);
        MavenCoordinates coordinates = (publishingExtension == null)
                ? mavenCoordinatesFromDependencyProject(project)
                : mavenCoordinatesFromPublications(project, publishingExtension);
        return AugmentedCoordinatesModuleDependency.augment(projectDependency, coordinates);
    }

    private MavenCoordinates mavenCoordinatesFromPublications(Project project, PublishingExtension publishingExtension) {
        PublicationContainer publications = publishingExtension.getPublications();
        MavenPublication mavenPublication = (MavenPublication) publications.iterator().next();
        logIfPublicationAmbiguous(project, publications, mavenPublication);
        return new MavenCoordinates(
                mavenPublication.getGroupId(),
                mavenPublication.getArtifactId(),
                mavenPublication.getVersion());
    }

    private void logIfPublicationAmbiguous(Project project, PublicationContainer publications, MavenPublication mavenPublication) {
        if (publications.size() > 1) {
            LOGGER.warn(
                "Plugin depends on project '{}', but project has multiple Maven publications. Will use first publication ({}) as dependency in POM.",
                project.getPath(),
                buildCoordinateString(mavenPublication)
            );
        }
    }

    private String buildCoordinateString(MavenPublication publication) {
        return publication.getGroupId() + ":" +
                publication.getArtifactId() + ":" +
                publication.getVersion();
    }

    private MavenCoordinates mavenCoordinatesFromDependencyProject(Project project) {
        return new MavenCoordinates(
                project.getGroup().toString(),
                project.getName(),
                project.getVersion().toString());
    }

    private boolean isDuplicateDependency(List<Dependency> mavenDependencies, ModuleDependency dependency) {
        for (Dependency mavenDependency : mavenDependencies) {
            if (mavenDependency.getGroupId().equals(dependency.getGroup())
                    && mavenDependency.getArtifactId().equals(dependency.getName())
                    && hasEqualOrNullVersions(mavenDependency, dependency)) {
                return true;
            }
        }
        return false;
    }

    private boolean hasEqualOrNullVersions(Dependency mavenDependency, ModuleDependency moduleDependency) {
        return mavenDependency.getVersion() ==  moduleDependency.getVersion();
    }

    private void addDependency(List<Dependency> mavenDependencies,
                               ModuleDependency moduleDependency, String scope) {
        Dependency mavenDependency = new Dependency();
        mavenDependency.setGroupId(moduleDependency.getGroup());
        mavenDependency.setArtifactId(moduleDependency.getName());
        mavenDependency.setVersion(moduleDependency.getVersion());
        mavenDependency.setScope(scope);
        logTransitiveDependency(moduleDependency);
        addExclusions(mavenDependency, moduleDependency);
        mavenDependencies.add(mavenDependency);
    }

    private void addExclusions(Dependency mavenDependency, ModuleDependency moduleDependency) {
        Set<ExcludeRule> excludeRules = moduleDependency.getExcludeRules();
        if (excludeRules != null) {
            for (ExcludeRule excludeRule : excludeRules) {
                Exclusion mavenExclusion = new Exclusion();
                mavenExclusion.setGroupId(excludeRule.getGroup());
                mavenExclusion.setArtifactId(excludeRule.getModule());
                mavenDependency.addExclusion(mavenExclusion);
            }
        }
    }

    private void logTransitiveDependency(ModuleDependency moduleDependency) {
        if (!moduleDependency.isTransitive()) {
            LOGGER.warn(
                    "Dependency {}:{}:{} is marked as non-transitive, " +
                            "but this is not supported by the underlying " +
                            "Maven publishing mechanism",
                    moduleDependency.getGroup(),
                    moduleDependency.getName(),
                    moduleDependency.getVersion());
        }
    }
}
