/*
 * Decompiled with CFR 0.152.
 */
package com.helger.base.charset;

import com.helger.annotation.Nonempty;
import com.helger.annotation.Nonnegative;
import com.helger.annotation.WillNotClose;
import com.helger.annotation.concurrent.Immutable;
import com.helger.annotation.style.CodingStyleguideUnaware;
import com.helger.annotation.style.ReturnsMutableCopy;
import com.helger.base.array.ArrayHelper;
import com.helger.base.charset.EUnicodeBOM;
import com.helger.base.enforce.ValueEnforcer;
import com.helger.base.io.iface.IHasInputStream;
import com.helger.base.io.nonblocking.NonBlockingPushbackInputStream;
import com.helger.base.io.stream.StreamHelper;
import com.helger.base.string.StringHelper;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.SortedMap;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Immutable
public final class CharsetHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(CharsetHelper.class);
    @CodingStyleguideUnaware
    private static final SortedMap<String, Charset> ALL_CHARSETS = Charset.availableCharsets();
    private static final CharsetHelper INSTANCE = new CharsetHelper();

    private CharsetHelper() {
    }

    public static @NonNull Charset getCharsetFromName(@Nonempty @NonNull String string) {
        ValueEnforcer.notNull(string, "CharsetName");
        try {
            return Charset.forName(string);
        }
        catch (IllegalCharsetNameException illegalCharsetNameException) {
            throw new IllegalArgumentException("Charset '" + string + "' unsupported in Java", illegalCharsetNameException);
        }
        catch (UnsupportedCharsetException unsupportedCharsetException) {
            throw new IllegalArgumentException("Charset '" + string + "' unsupported on this platform", unsupportedCharsetException);
        }
    }

    public static @Nullable Charset getCharsetFromNameOrNull(@Nullable String string) {
        return CharsetHelper.getCharsetFromNameOrDefault(string, null);
    }

    public static @Nullable Charset getCharsetFromNameOrDefault(@Nullable String string, @Nullable Charset charset) {
        if (StringHelper.isNotEmpty(string)) {
            try {
                return CharsetHelper.getCharsetFromName(string);
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
        }
        return charset;
    }

    @ReturnsMutableCopy
    public static @NonNull Map<String, Charset> getAllCharsets() {
        return new LinkedHashMap<String, Charset>(ALL_CHARSETS);
    }

    public static @Nullable String getAsStringInOtherCharset(@Nullable String string, @NonNull Charset charset, @NonNull Charset charset2) {
        ValueEnforcer.notNull(charset, "CurrentCharset");
        ValueEnforcer.notNull(charset2, "NewCharset");
        if (string == null || charset.equals(charset2)) {
            return string;
        }
        return new String(string.getBytes(charset), charset2);
    }

    @Nonnegative
    public static int getUTF8ByteCount(@Nullable String string) {
        return string == null ? 0 : CharsetHelper.getUTF8ByteCount(string.toCharArray());
    }

    @Nonnegative
    public static int getUTF8ByteCount(@Nullable char[] cArray) {
        int n = 0;
        if (cArray != null) {
            for (char c : cArray) {
                n += CharsetHelper.getUTF8ByteCount(c);
            }
        }
        return n;
    }

    @Nonnegative
    public static int getUTF8ByteCount(char c) {
        return CharsetHelper.getUTF8ByteCount((int)c);
    }

    @Nonnegative
    public static int getUTF8ByteCount(@Nonnegative int n) {
        ValueEnforcer.isBetweenInclusive(n, "c", 0, 65535);
        if (n == 0) {
            return 2;
        }
        if (n <= 127) {
            return 1;
        }
        if (n <= 2047) {
            return 2;
        }
        if (n <= 55295) {
            return 3;
        }
        return 0;
    }

    public static @NonNull InputStreamAndCharset getInputStreamAndCharsetFromBOM(@WillNotClose @NonNull InputStream inputStream) {
        ValueEnforcer.notNull(inputStream, "InputStream");
        int n = EUnicodeBOM.getMaximumByteCount();
        NonBlockingPushbackInputStream nonBlockingPushbackInputStream = new NonBlockingPushbackInputStream(StreamHelper.getBuffered(inputStream), n);
        try {
            byte[] byArray = new byte[n];
            int n2 = nonBlockingPushbackInputStream.read(byArray);
            EUnicodeBOM eUnicodeBOM = null;
            Charset charset = null;
            if (n2 > 0) {
                eUnicodeBOM = EUnicodeBOM.getFromBytesOrNull(ArrayHelper.getCopy(byArray, 0, n2));
                if (eUnicodeBOM == null) {
                    nonBlockingPushbackInputStream.unread(byArray, 0, n2);
                } else {
                    int n3;
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("Found " + String.valueOf((Object)eUnicodeBOM) + " on " + inputStream.getClass().getName());
                    }
                    if ((n3 = eUnicodeBOM.getByteCount()) < n2) {
                        nonBlockingPushbackInputStream.unread(byArray, n3, n2 - n3);
                    }
                    charset = eUnicodeBOM.getCharset();
                }
            }
            return new InputStreamAndCharset(nonBlockingPushbackInputStream, eUnicodeBOM, charset);
        }
        catch (IOException iOException) {
            LOGGER.error("Failed to determine BOM", (Throwable)iOException);
            StreamHelper.close(nonBlockingPushbackInputStream);
            throw new UncheckedIOException(iOException);
        }
    }

    public static @NonNull InputStreamReader getReaderByBOM(@NonNull InputStream inputStream, @NonNull Charset charset) {
        ValueEnforcer.notNull(inputStream, "InputStream");
        ValueEnforcer.notNull(charset, "FallbackCharset");
        InputStreamAndCharset inputStreamAndCharset = CharsetHelper.getInputStreamAndCharsetFromBOM(inputStream);
        Charset charset2 = inputStreamAndCharset.getCharset(charset);
        return StreamHelper.createReader(inputStreamAndCharset.getInputStream(), charset2);
    }

    public static final class InputStreamAndCharset
    implements IHasInputStream {
        private final InputStream m_aIS;
        private final EUnicodeBOM m_eBOM;
        private final Charset m_aCharset;

        public InputStreamAndCharset(@NonNull InputStream inputStream, @Nullable EUnicodeBOM eUnicodeBOM, @Nullable Charset charset) {
            this.m_aIS = inputStream;
            this.m_eBOM = eUnicodeBOM;
            this.m_aCharset = charset;
        }

        @Override
        public @NonNull InputStream getInputStream() {
            return this.m_aIS;
        }

        @Override
        public boolean isReadMultiple() {
            return false;
        }

        public @Nullable EUnicodeBOM getBOM() {
            return this.m_eBOM;
        }

        public boolean hasBOM() {
            return this.m_eBOM != null;
        }

        public @Nullable Charset getCharset() {
            return this.m_aCharset;
        }

        public boolean hasCharset() {
            return this.m_aCharset != null;
        }

        public @Nullable Charset getCharset(@Nullable Charset charset) {
            return this.m_aCharset != null ? this.m_aCharset : charset;
        }
    }
}

