package com.xebialabs.deployit.core.rest.api

import java.io._
import java.nio.file.{Files, Path, Paths}
import java.util.{Collections, UUID}

import com.xebialabs.deployit.core.api.DownloadService
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import grizzled.slf4j.Logging
import javax.ws.rs.PathParam
import javax.ws.rs.core.{Response, StreamingOutput}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Controller

import scala.collection.convert.ImplicitConversionsToScala._
import scala.util.Try

object DownloadResource {
  val CONTENT_TYPES_FOLDERNAME = "content-types"
}

@Controller
class DownloadResource @Autowired()(@Qualifier("exportRootPath") exportRootPath: String)
  extends AbstractSecuredResource with DownloadService with Logging {

  private val exportRoot = new File(exportRootPath).ensurePresent(orElse = s"Cannot create download cache at $exportRootPath")

  private val contentTypesRepo = new File(exportRoot, DownloadResource.CONTENT_TYPES_FOLDERNAME)
    .ensurePresent(orElse = s"Cannot create content-type repository under ${exportRoot.getAbsolutePath}")


  override def download(@PathParam("downloadkey") downloadKey: String): Response = {
    val downloadFolder = downloadFolderFor(downloadKey).toFile

    val contents: File = Try(downloadFolder.listFiles().head).getOrElse(
      throw new NotFoundException(s"The download with key [$downloadKey] was not found.")
    )
    val contentType: String = Try(Files.readAllLines(contentTypeFor(downloadKey)).head).getOrElse(
      throw new NotFoundException(s"Cannot determine content type for download with key [$downloadKey]")
    )
    val output: StreamingOutput = contents.toStreamingOutput {
      LocalFile.from(downloadFolder).deleteRecursively()
      contentTypeFor(downloadKey).toFile.delete()
    }

    info(s"Going to stream download of [$downloadKey] with type [$contentType]")
    Response.ok(output, contentType)
      .header("Content-Disposition", s"""attachment; filename="${contents.getName}"""")
      .header("Content-Length", String.valueOf(contents.length)).build
  }

  def register(file: File, contentType: String): String = {
    val downloadKey: String = UUID.randomUUID.toString
    val downloadFolder: Path = downloadFolderFor(downloadKey)
    if (!downloadFolder.toFile.mkdirs) {
      throw new DeployitException(s"Cannot create directory for download: $downloadFolder")
    }
    val movedFile = file.moveTo(downloadFolder)

    info(s"Registering download [$movedFile] with content type [$contentType] and key [$downloadKey]")
    registerContentType(downloadKey, contentType)
    downloadKey
  }

  private def downloadFolderFor(downloadKey: String): Path = Paths.get(exportRoot.getAbsolutePath, downloadKey)

  private def registerContentType(downloadKey: String, contentType: String) = {
    val registerPath = contentTypeFor(downloadKey)
    try {
      Files.write(registerPath, Collections.singletonList(contentType))
    } catch {
      case e: IOException =>
        throw new DeployitException(e,
          s"Cannot register content type for download key [$downloadKey] in [$registerPath]")
    }
  }

  private def contentTypeFor(downloadKey: String): Path = Paths.get(contentTypesRepo.getAbsolutePath, downloadKey)


  private implicit class FileUtils(file: File) {

    def toStreamingOutput(finalizeWrite: => Unit): StreamingOutput = (outputStream: OutputStream) => {
      val is: InputStream = new FileInputStream(file)
      try {
        OverthereUtils.write(is, outputStream)
      } finally {
        if (is != null) is.close()
        finalizeWrite
      }
    }

    def moveTo(destinationFolder: Path): Path = {
      val destination = destinationFolder.resolve(file.getName)
      try {
        Files.move(file.toPath, destination)
      } catch {
        case e: IOException =>
          throw new DeployitException(e,
            s"Could not move file ${file.getAbsolutePath} to download cache [$destination]"
          )
      }
    }

    def ensurePresent(orElse: String): File = if (file.exists() || file.mkdirs())
      file
    else
      throw new DeployitException(orElse)
  }

}
