package com.xebialabs.xlplatform.config

import com.typesafe.config.ConfigFactory.{parseReader, parseResources}
import com.typesafe.config._
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.xlplatform.utils.ClassLoaderUtils._
import grizzled.slf4j.Logging
import org.springframework.util.ResourceUtils

import java.io.{File, InputStream, InputStreamReader}
import java.nio.file.Files
import scala.jdk.CollectionConverters._
import scala.util.Try

object ConfigLoader extends Logging {

  type FindResourceFile = String => Option[File]

  def loadWithDynamic(configResource: String): Config = {
    loadWithDynamic(parseResources(configResource))
  }

  def loadWithDynamic(inputStream: InputStream): Config = {
    loadWithDynamic(parseReader(new InputStreamReader(inputStream)))
  }

  def loadWithDynamic(config: Config, defaultReference: Config = ConfigFactory.defaultReference()): Config = {
    resolveWithDynamic(config.withFallback(defaultReference))
  }

  def loadSecuredWithDynamic(
                              configResource: String,
                              keyHolder: SecretKeyHolder,
                              findResourceFile: FindResourceFile = findFirstInClassPath,
                              defaultReference: Config = ConfigFactory.defaultReference()
                            ): Config = {
    replacePlainTextIfNeeded(configResource, keyHolder, findResourceFile)
    loadWithDynamic(parseResources(configResource), defaultReference).decrypted(keyHolder)
  }

  def loadFromInputSecuredWithDynamic(
                                       inputStream: InputStream,
                                       keyHolder: SecretKeyHolder,
                                       defaultReference: Config = ConfigFactory.defaultReference()
                                     ): Config = {
    val config = parseReader(new InputStreamReader(inputStream))
    loadWithDynamic(config, defaultReference).decrypted(keyHolder)
  }

  private def replacePlainTextIfNeeded(configResource: String, keyHolder: SecretKeyHolder, findResourceFile: FindResourceFile): Unit = {
    val config = parseResources(configResource, ConfigParseOptions.defaults().setAllowMissing(true))

    if (config.hasPlainPasswords(keyHolder)) {
      warn(s"Configuration file $configResource contains plain text passwords")
      findResourceFile(configResource) match {
        case Some(file) =>
          info(s"Writing secured configuration file to ${file.getAbsolutePath}")
          Files.write(file.toPath, Seq(config.encrypted(keyHolder).resolve(ConfigResolveOptions.defaults().setAllowUnresolved(true)).root().render(ConfigRenderOptions.defaults.setFormatted(true).setJson(false).setOriginComments(false).setComments(true))).asJava)
        case None =>
          warn(s"Configuration file $configResource cannot be found in file system. Encryption aborted!")
      }
    }
  }

  private def resolveWithDynamic(config: Config) = {
    val serverConfig = if (config.hasPath("xl.conf-file.location")) {
      val serverConfigLocation = config.getString("xl.conf-file.location")
      val serverConfigFile = ResourceUtils.getFile(serverConfigLocation)
      ConfigFactory.parseFile(serverConfigFile, ConfigParseOptions.defaults().setSyntax(ConfigSyntax.PROPERTIES))
    } else {
      ConfigFactory.empty()
    }

    parseResources("dynamic-reference.conf").withFallback(config).withFallback(serverConfig).resolve()
  }

  private def findFirstInClassPath(resource: String): Option[File] = {
    def fileExists(maybeFile: Option[File]): Boolean = {
      maybeFile.exists(file => file.exists())
    }

    classLoader.getResources(resource).asScala.map(url => Try(new File(url.toURI)).toOption).find(fileExists).flatten
  }

}
