package com.xebialabs.deployit.util;

import java.io.*;
import java.util.Properties;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class PropertyUtil {

    private static final String HEADER_COMMENT = "XL Deploy configuration file.\n\nNotes: \n"
            + " * If you modify this file, you must restart the XL Deploy server.\n"
            + " * Use `repository.keystore.password` property to specify the password used to open the repository keystore file, \n"
            + "   which contains the encryption key for the rest of the passwords.\n"
            + " * The other passwords specified in this file are normally encrypted, but you can change their value using plaintext \n"
            + "   and at the next start of the server the modified passwords will be re-encrypted.\n";

    public static Properties readPropertiesFile(File file) {
        Properties properties = new Properties();
        try (InputStream inputStream = new FileInputStream(file)) {
            properties.load(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return properties;
    }

    public static void storePropertiesFile(File file, Properties properties) {
        try (BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(file))) {
            properties.store(output, HEADER_COMMENT);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static int parseIntSilent(Properties properties, String KEY, int defaultValue) {
        try {
            String value = properties.getProperty(KEY);
            return value != null ? Integer.parseInt(value) : defaultValue;
        } catch (NumberFormatException nfe) {
            return defaultValue;
        }
    }

    public static double parseDoubleSilent(Properties properties, String KEY, double defaultValue) {
        try {
            String value = properties.getProperty(KEY);
            return value != null ? Double.parseDouble(value) : defaultValue;
        } catch (NumberFormatException nfe) {
            return defaultValue;
        }
    }

    public static long parseLongSilent(Properties properties, String KEY, long defaultValue) {
        try {
            String value = properties.getProperty(KEY);
            return value != null ? Long.parseLong(value) : defaultValue;
        } catch (NumberFormatException nfe) {
            return defaultValue;
        }
    }

    public static void updateProperty(Properties properties, String key, String value, String initialValue, Consumer<String> setter) {
        updateProperty(properties, key, () -> value, () -> initialValue, setter);
    }

    public static void updateProperty(Properties properties, String key, String value, Consumer<String> setter) {
        updateProperty(properties, key, () -> value, () -> null, setter);
    }

    public static void updateProperty(Properties properties, String key, Supplier<String> getter, Supplier<String> initialValueSupplier, Consumer<String> setter) {
        String value = getter.get();
        if (value == null) {
            String initialValue = initialValueSupplier.get();
            setter.accept(initialValue);
            if (initialValue == null) {
                properties.remove(key);
            } else {
                properties.setProperty(key, initialValue);
            }
        } else {
            properties.setProperty(key, value);
        }
    }

    public static void updateBooleanProperty(Properties properties, String key, Boolean value, Boolean initialValue, Consumer<Boolean> setter) {
        updateBooleanProperty(properties, key, () -> value, () -> initialValue, setter);
    }

    private static void updateBooleanProperty(Properties properties, String key, Supplier<Boolean> getter, Supplier<Boolean> initialValueSupplier, Consumer<Boolean> setter) {
        Boolean value = getter.get();
        if (value == null) {
            Boolean initialValue = initialValueSupplier.get();
            setter.accept(initialValue);
            properties.setProperty(key, initialValue.toString());
        } else {
            properties.setProperty(key, value.toString());
        }
    }

    public static void updateIntProperty(Properties properties, String key, Integer value, Integer initialValue, Consumer<Integer> setter) {
        updateIntProperty(properties, key, () -> value, () -> initialValue, setter);
    }

    private static void updateIntProperty(Properties properties, String key, Supplier<Integer> getter, Supplier<Integer> initialValueSupplier, Consumer<Integer> setter) {
        Integer value = getter.get();
        if (value == null) {
            Integer initialValue = initialValueSupplier.get();
            setter.accept(initialValue);
            properties.setProperty(key, initialValue.toString());
        } else {
            properties.setProperty(key, value.toString());
        }
    }

    public static void updateLongProperty(Properties properties, String key, Long value, Long initialValue, Consumer<Long> setter) {
        updateLongProperty(properties, key, () -> value, () -> initialValue, setter);
    }

    private static void updateLongProperty(Properties properties, String key, Supplier<Long> getter, Supplier<Long> initialValueSupplier, Consumer<Long> setter) {
        Long value = getter.get();
        if (value == null) {
            Long initialValue = initialValueSupplier.get();
            setter.accept(initialValue);
            properties.setProperty(key, initialValue.toString());
        } else {
            properties.setProperty(key, value.toString());
        }
    }

    public static void updateDoubleProperty(Properties properties, String key, Double value, Double initialValue, Consumer<Double> setter) {
        updateDoubleProperty(properties, key, () -> value, () -> initialValue, setter);
    }

    private static void updateDoubleProperty(Properties properties, String key, Supplier<Double> getter, Supplier<Double> initialValueSupplier, Consumer<Double> setter) {
        Double value = getter.get();
        if (value == null) {
            Double initialValue = initialValueSupplier.get();
            setter.accept(initialValue);
            properties.setProperty(key, initialValue.toString());
        } else {
            properties.setProperty(key, value.toString());
        }
    }

}
