package com.xebialabs.deployit.upgrade;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.booter.local.PluginVersions;
import com.xebialabs.deployit.server.api.upgrade.RepositoryInitialization;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService;
import com.xebialabs.xlplatform.upgrade.UpgraderHelper;

import static java.lang.String.format;

@Component
public class RepositoryInitializer implements IRepositoryInitializer, ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(RepositoryInitializer.class);

    private ApplicationContext context;
    private final InitializationStrategy initializationStrategy;
    private final RepositoryVersionService repositoryVersionService;

    @Autowired
    public RepositoryInitializer(InitializationStrategy initializationStrategy, RepositoryVersionService repositoryVersionService) {
        this.initializationStrategy = initializationStrategy;
        this.repositoryVersionService = repositoryVersionService;
    }

    public void initializeComponents() {
        Map<String, List<RepositoryInitialization>> initializations = getInitializations();
        Map<String, String> versions = repositoryVersionService.getAllComponentVersions();

        for (String component : getRegisteredPlugins()) {
            initializeComponent(component, versions, initializations);
        }
    }

    private void initializeComponent(final String component, final Map<String, String> versions,
                                     final Map<String, List<RepositoryInitialization>> initializations) {
        String componentVersion = versions.get(component);

        try {
            if(componentVersion != null) {
                Version.valueOf(component, componentVersion);
            }
        } catch (RuntimeException e) {
            logger.warn(format("Component [%s] has an invalid version", component), e);
            return;
        }

        if (componentVersion == null) {
            doInitializeComponent(component, initializations);

            repositoryVersionService.storeVersionOfComponent(Version.valueOf(component, Version.VERSION_0));
        } else {
            logger.debug("Component {} is already initialized", component);
        }
    }

    private void doInitializeComponent(final String component, final Map<String, List<RepositoryInitialization>> initializations) {
        List<RepositoryInitialization> componentInitializations = initializations.get(component);
        if (componentInitializations != null && !componentInitializations.isEmpty()) {
            final List<RepositoryInitialization> sortedInitializations = new ArrayList<>(componentInitializations);
            sortedInitializations.sort(Comparator.comparing(o -> o.getClass().getSimpleName()));

            logger.info("Initializing repository for component [{}] using: {}", component, sortedInitializations);
            initializationStrategy.doInitializations(sortedInitializations);
        }
    }

    private Map<String, List<RepositoryInitialization>> getInitializations() {
        Collection<RepositoryInitialization> allInitializations = findAndInstantiateApplicableInitializers();
        logger.debug("Found the following initializations: [{}]", allInitializations);
        Map<String, List<RepositoryInitialization>> map = new HashMap<>();
        for (RepositoryInitialization init : allInitializations) {
            map.computeIfAbsent(init.getComponent(), i -> new ArrayList<>()).add(init);
        }
        return map;
    }

    Collection<RepositoryInitialization> findAndInstantiateApplicableInitializers() {
        Set<Class<? extends RepositoryInitialization>> initializerClasses = initializationStrategy.findApplicableInitializerTypes();
        return UpgraderHelper.instantiateClasses(initializerClasses, context);
    }

    Set<String> getRegisteredPlugins() {
        return PluginVersions.getRegisteredPlugins();
    }

    @Override
    public void setApplicationContext(final ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }
}
