package com.xebialabs.xlplatform.artifact.resolution.http

import com.typesafe.config.Config
import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver
import com.xebialabs.deployit.engine.spi.artifact.resolution.{ArtifactResolver, ResolvedArtifactFile}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.credentials.{Credentials, UsernamePasswordCredentials => ArtifactServerCredentials}
import com.xebialabs.deployit.plugin.proxy.{ProxyServer, ProxyServerType}
import com.xebialabs.xlplatform.config.{ConfigLoader, ConfigurationHolder}
import grizzled.slf4j.Logging
import org.apache.http.auth._
import org.apache.http.client.methods.{CloseableHttpResponse, HttpGet, HttpHead, HttpUriRequest}
import org.apache.http.client.protocol.HttpClientContext
import org.apache.http.config.RegistryBuilder
import org.apache.http.conn.HttpHostConnectException
import org.apache.http.conn.socket.ConnectionSocketFactory
import org.apache.http.conn.ssl._
import org.apache.http.impl.auth.BasicScheme
import org.apache.http.impl.client._
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager
import org.apache.http.protocol.HttpContext
import org.apache.http.ssl.SSLContextBuilder
import org.apache.http.{HttpHost, HttpResponse, HttpStatus}

import java.io.{ByteArrayInputStream, FileNotFoundException, InputStream}
import java.net.{InetSocketAddress, Proxy, Socket, URI, UnknownHostException}
import scala.util.Try

object HttpArtifactResolver {
  val Protocols: Array[String] = Array("http", "https")

  private[http] val ContentDispositionHeader = "Content-Disposition"

  private[http] val ContentDispositionFileName = """.*;[ ]*filename="?([^;^"]*)"?.*""".r
}

@Resolver(protocols = Array("http", "https"))
class HttpArtifactResolver extends ArtifactResolver with Logging {

  import com.xebialabs.xlplatform.artifact.resolution.http.HttpArtifactResolver._

  override def resolveLocation(artifact: SourceArtifact): ResolvedArtifactFile = new ResolvedArtifactFile {

    private var fileName: Option[String] = None
    private var fileSize: Long = -1L
    private var openConnections: Seq[CloseableHttpClient] = Seq()

    override def getFileName: String = fileName.getOrElse {
      fileName = filenameOption(new HttpHead(artifact.getFileUri))
        .orElse(filenameOption(new HttpGet(artifact.getFileUri)))
        .orElse(getFilenameFromUri)
      fileName.get
    }
    override def getFileSize: Long = fileSize
    override def close(): Unit = openConnections.foreach(_.close())

    override def openStream: InputStream = {
      val uri = getFileUri(artifact)

      val builder: HttpClientBuilder = HttpClientBuilder.create().useSystemProperties()
      val context: HttpClientContext = setContextForAuthentication(uri)

      Option(artifact.getProxySettings).map(_.asInstanceOf[ProxyServer]).foreach(proxyServer =>
          proxyServer.getProtocol match {
            case ProxyServerType.SOCKS =>
              applySocksProxyConfiguration(builder, context, proxyServer)
            case ProxyServerType.HTTP =>
              applyHttpProxyConfiguration(builder, context, proxyServer)
            case ProxyServerType.HTTPS =>
              applyHttpProxyConfiguration(builder, context, proxyServer)
              applyHttpsProxyConfiguration(builder)
        }
      )

      builder.setDefaultCredentialsProvider(context.getCredentialsProvider)
      val client = builder.build()

      val response = doRequest(client, context, new HttpGet(uri))
      // Extract file size from Content-Length using HttpEntity
      if (response.getEntity != null) {
        val contentLength = response.getEntity.getContentLength
        if (contentLength > 0) {
          fileSize = contentLength
          logger.debug(s"Extracted file size from HttpEntity.getContentLength: $fileSize bytes for ${artifact.getFileUri}")
        }
      }
      response.getStatusLine.getStatusCode match {
        case sc if isSuccessfulStatus(sc) => response.getEntity.getContent
        case HttpStatus.SC_NOT_FOUND if httpSettings.ignoreMissingArtifact =>
          logger.info(s"ignoreMissingArtifact mode: Could not find file ${artifact.getFileUri} on the http server, creating an empty input stream.")
          new ByteArrayInputStream(s"Could not find artifact data for ${artifact.getFileUri}.".getBytes("UTF-8"))
        case _ => throw new RuntimeException(s"Server returned non-successful status code ${response.getStatusLine.getStatusCode} when accessing ${artifact.getFileUri}")
      }
    }

    private def applySocksProxyConfiguration(builder: HttpClientBuilder, context: HttpClientContext, proxyServer: ProxyServer): Unit = {
      val reg = RegistryBuilder.create[ConnectionSocketFactory]
        .register("http", new SocksConnectionSocketFactory)
        .register("https", new SocksConnectionSocketFactory).build
      val connectionManager = new PoolingHttpClientConnectionManager(reg)
      builder.setConnectionManager(connectionManager)
      val socksAddress = new InetSocketAddress(proxyServer.getHostname, proxyServer.getPort)
      context.setAttribute("socks.address", socksAddress)
    }

    private def applyHttpsProxyConfiguration(builder: HttpClientBuilder): Unit = {
      val sslContext = new SSLContextBuilder().loadTrustMaterial(null, new TrustSelfSignedStrategy).build
      builder.setSSLContext(sslContext).setSSLHostnameVerifier(new NoopHostnameVerifier())
    }

    private def applyHttpProxyConfiguration(builder: HttpClientBuilder, context: HttpClientContext, proxyServer: ProxyServer): Unit = {
      val proxyHost = new HttpHost(proxyServer.getHostname, proxyServer.getPort, proxyServer.getProtocol.getValue)
      builder.setProxy(proxyHost)
      if (proxyServer.getUsername != null) {
        builder.setProxyAuthenticationStrategy(new ProxyAuthenticationStrategy)
        if (context.getCredentialsProvider == null) {
          context.setCredentialsProvider(new BasicCredentialsProvider)
        }
        context.getCredentialsProvider.setCredentials(
          new AuthScope(proxyServer.getHostname, proxyServer.getPort), new UsernamePasswordCredentials(proxyServer.getUsername, proxyServer.getPassword))

        if (context.getAuthCache == null) {
          context.setAuthCache(new BasicAuthCache())
        }
        context.getAuthCache.put(proxyHost, new BasicScheme(ChallengeState.PROXY))
      }
    }

    def doRequest(client: CloseableHttpClient, context: HttpClientContext, request: HttpGet): CloseableHttpResponse = {
      try {
        val response = client.execute(request, context)
        fileName = fileName.orElse(filenameFromHeader(response))
        openConnections :+= client
        response
      } catch {
        case e: UnknownHostException => throw new RuntimeException(s"Host can not be found", e)
        case e: FileNotFoundException => throw new RuntimeException(s"File not found: ${artifact.getFileUri}", e)
        case e: HttpHostConnectException => throw new RuntimeException(s"Proxy connection error", e)
        case e: Throwable => throw new RuntimeException(s"Erorr downloading artifact: ${artifact.getFileUri}", e)
      }
    }

    private def resolveCredentials(uri: URI, credentials: Credentials): Option[(String, String)] = {
      val userInfo = """(\w+):(\w+)""".r

      (Option(credentials), Option(uri.getUserInfo)) match {
        case (Some(_), Some(_)) =>
          throw new IllegalArgumentException(s"Can not use both style of credentials: ${uri.toString}")
        case (Some(cred: ArtifactServerCredentials), _) =>
          Some((cred.getUsername, cred.getPassword))
        case (None, Some(userInfo(username, password))) =>
          Some((username, password))
        case _ => None
      }
    }

    private def setContextForAuthentication(uri: URI) = {
      val context = HttpClientContext.create
      val credentials = resolveCredentials(uri, artifact.getCredentials)
      credentials.foreach {
        case (userName, password) =>
          cacheCredentialsInContext(context, userName, password, new HttpHost(uri.getHost, uri.getPort, uri.getScheme))
      }
      context
    }

    private def cacheCredentialsInContext(context: HttpClientContext, user: String, password: String, httpHost: HttpHost): Unit = {
      val credentialsProvider = new BasicCredentialsProvider
      val authCache = new BasicAuthCache

      credentialsProvider.setCredentials(new AuthScope(httpHost.getHostName, httpHost.getPort), new UsernamePasswordCredentials(user, password))
      context.setCredentialsProvider(credentialsProvider)

      authCache.put(httpHost, new BasicScheme)
      context.setAuthCache(authCache)
    }

    private def filenameFromHeader(response: HttpResponse): Option[String] = {
      Option(response.getLastHeader(ContentDispositionHeader)).map(_.getValue).collect {
        case ContentDispositionFileName(filename) => filename
      }.orElse(getFilenameFromUri)
    }

    private def filenameOption(request: HttpUriRequest): Option[String] = {
      val client = HttpClients.createDefault
      try {
        Try(client.execute(request)).toOption.collect {
          case response if isSuccessfulStatus(response.getStatusLine.getStatusCode) =>
            filenameFromHeader(response).get
        }
      } finally {
        client.close()
      }
    }

    private def getFilenameFromUri: Option[String] = Some(getFileUri(artifact).getSchemeSpecificPart.split('/').last)

    private def isSuccessfulStatus(sc: Int): Boolean = sc >= HttpStatus.SC_OK && sc < HttpStatus.SC_MULTIPLE_CHOICES
  }

  def getFileUri(artifact: SourceArtifact): URI = {
    URI.create(artifact.getFileUri)
  }

  override def validateCorrectness(artifact: SourceArtifact): Boolean = Protocols.contains(getFileUri(artifact).getScheme)

  def httpSettings = new HttpSettings(ConfigLoader.loadWithDynamic(ConfigurationHolder.get()))
}

class HttpSettings(config: Config) {
  lazy val ignoreMissingArtifact: Boolean = config.getBoolean("xl.artifact.resolver.http.ignoreMissingArtifact")
}

class SocksConnectionSocketFactory extends ConnectionSocketFactory {
  override def createSocket(context: HttpContext): Socket = {
    val socksAddress = context.getAttribute("socks.address").asInstanceOf[InetSocketAddress]
    val proxy = new Proxy(Proxy.Type.SOCKS, socksAddress)
    new Socket(proxy)
  }

  override def connectSocket(connectTimeout: Int, socket: Socket, host: HttpHost, remoteAddress: InetSocketAddress, localAddress: InetSocketAddress, context: HttpContext): Socket = {
    val sock: Socket = if (socket != null) {
      socket
    }
    else {
      createSocket(context)
    }
    if (localAddress != null) {
      sock.bind(localAddress)
    }
    sock.connect(remoteAddress, connectTimeout)
    sock
  }

}
