package com.xebialabs.deployit.upgrade;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import com.xebialabs.deployit.booter.local.PluginVersions;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.event.ShutdownEvent;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService;
import com.xebialabs.xlplatform.upgrade.UpgraderHelper;

import static java.lang.String.format;

public class Upgrader implements ApplicationContextAware {

    private static final Logger logger = LoggerFactory.getLogger(Upgrader.class);

    private final Set<String> components = new LinkedHashSet<>(); // Must be ordered, but duplicates should be filtered out
    private final String COMPONENT_DEPLOYIT = "deployit";
    private final Map<String, List<Upgrade>> upgrades = new HashMap<>();
    private final boolean forceUpgrades;

    private ApplicationContext applicationContext;
    private final UpgradeStrategy upgradeStrategy;
    private boolean questionAsked = false;
    private final RepositoryVersionService repositoryVersionService;

    public Upgrader(UpgradeStrategy upgradeStrategy, boolean forceUpgrades, RepositoryVersionService repositoryVersionService) {
        this.upgradeStrategy = upgradeStrategy;
        this.forceUpgrades = forceUpgrades;
        this.repositoryVersionService = repositoryVersionService;
    }

    public void addComponent(String component) {
        components.add(component);
    }

    public void addUpgrade(Upgrade upgrade) {
        String component = upgrade.upgradeVersion().getComponent();
        addComponent(component);
        upgrades.computeIfAbsent(component, c -> new ArrayList<>()).add(upgrade);
    }

    public void applyUpgrades() {
        applyUpgrades(false);
    }

    public void applyUpgrades(boolean verifyOnly) {
        components.forEach(component -> this.upgradeComponent(component, verifyOnly));
    }

    public void autoUpgrade() {
        autoUpgrade(false);
    }

    public void autoUpgrade(boolean verifyOnly) {
        PluginVersions.getRegisteredPlugins()
                .stream()
                .sorted((left, right) -> COMPONENT_DEPLOYIT.equals(left) ? -1 : 0)
                .forEach(components::add);
        findUpgrades().forEach((c, l) -> upgrades.computeIfAbsent(c, i -> new ArrayList<>()).addAll(l));
        applyUpgrades(verifyOnly);
    }

    public Version getComponentVersion(String component) {
        return repositoryVersionService.readVersionOfComponent(component);
    }

    void upgradeComponent(String component) {
        upgradeComponent(component, false);
    }

    void upgradeComponent(String component, boolean verifyOnly) {
        logger.debug("Checking component [{}] for upgrades", component);

        Version componentVersion;
        try {
            componentVersion = getComponentVersion(component);
        } catch (RuntimeException e) {
            logger.warn(format("Component [%s] has an invalid version -- skipping upgrade", component), e);
            return;
        }

        if (componentVersion == null) {
            logger.debug("Skipping upgrade of component {} as it is not registered", component);
            return;
        }

        logger.debug("Component [{}] has version [{}] in the repository", component, componentVersion);
        List<Upgrade> componentUpgrades = Optional.ofNullable(this.upgrades.get(component)).orElse(new ArrayList<>());
        List<Upgrade> applicableUpgrades = UpgraderHelper.filterApplicable(componentUpgrades, componentVersion);

        if (componentVersion.isVersion0()) {
            if (verifyOnly) {
                throw new UpgradeRejectedException("Verification failed, found new component.");
            }
            Version versionOfComponent = UpgraderHelper.findRecentVersion(applicableUpgrades, component);
            if (versionOfComponent == null && PluginVersions.getVersionFor(component) != null) {
                versionOfComponent = Version.valueOf(component, PluginVersions.getVersionFor(component));
            }
            repositoryVersionService.storeVersionOfComponent(versionOfComponent);
            logger.debug("Setting version [{}] for component [{}]", versionOfComponent, component);
            return;
        }

        if (!applicableUpgrades.isEmpty()) {
            logger.info("Checked component [{}] which is at version [{}] -> Found upgrades to run: {}", component, componentVersion, applicableUpgrades);
            if (verifyOnly) {
                throw new UpgradeRejectedException("Verification failed, found new upgrades to apply.");
            }
            if (applicableUpgrades.stream().anyMatch(Upgrade::isAskForUpgrade)) {
                askForUpgrade();
            }
            applyUpgrades(applicableUpgrades);
        }
    }

    public void askForUpgrade() {
        if (!questionAsked && !forceUpgrades) {
            logger.warn("Ensure that you're running in 'interactive' mode and look at your console to continue the upgrade process.");
            String NL = System.lineSeparator();
            logger.info("Upgraders need to be run, asking user to confirm.");
            String msg = NL + "*** WARNING ***"
                    + NL + "We detected that we need to upgrade your repository"
                    + NL + "Before continuing we suggest you backup your repository in case the upgrade fails."
                    + NL + "Please ensure you have 'INFO' level logging configured."
                    + NL + "Please enter 'yes' if you want to continue [no]: ";
            System.out.print(msg);
            String response = read();
            if (!"yes".equalsIgnoreCase(response)) {
                logger.error("Did not receive an affirmative response on running upgrades, shutting down.");
                EventBusHolder.publish(new ShutdownEvent());
                throw new UpgradeRejectedException("Did not receive an affirmative response on running upgrades, shutting down.");
            }
            logger.info("User response was: {}", response);
            questionAsked = true;
        }
    }

    private void applyUpgrades(final List<Upgrade> applicableUpgrades) {
        Set<Version> versions = applicableUpgrades.stream().map(Upgrade::upgradeVersion).collect(Collectors.toCollection(TreeSet::new));
        Map<Version, List<Upgrade>> versionToUpgradeMap = new HashMap<>();
        applicableUpgrades.forEach(u -> versionToUpgradeMap.computeIfAbsent(u.upgradeVersion(), v -> new ArrayList<>()).add(u));

        for (final Version version : versions) {
            updateVersion(version, versionToUpgradeMap.get(version));
        }
    }

    protected void updateVersion(final Version version, final List<Upgrade> upgrades) {
        logger.info("Upgrading to version [{}]", version);
        upgradeStrategy.doUpgrades(upgrades);
        repositoryVersionService.storeVersionOfComponent(version);
    }

    public Map<String, List<Upgrade>> findUpgrades() {
        Set<Class<? extends Upgrade>> upgradeClasses = upgradeStrategy.findApplicableUpgradeTypes();
        Set<Upgrade> allUpgrades = UpgraderHelper.instantiateClasses(upgradeClasses, applicationContext);
        logger.debug("Found the following upgrades: [{}]", allUpgrades);
        Map<String, List<Upgrade>> byComponent = new HashMap<>();
        allUpgrades.forEach(u -> byComponent.computeIfAbsent(u.upgradeVersion().getComponent(), c -> new ArrayList<>()).add(u));
        return byComponent;
    }

    protected String read() {
        BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
        try {
            final String line = stdin.readLine();
            if (line != null) {
                return line.trim();
            } else {
                return null;
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void setApplicationContext(final ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
