package com.xebialabs.deployit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.Properties;

public final class ServerConfigFile {

    private final File configFile;

    public ServerConfigFile(File file) {
        configFile = file;
    }

    public boolean exists() {
        return configFile.exists();
    }

    public String getName() {
        return configFile.getName();
    }

    public void readIntoConfiguration(ServerConfiguration serverConfiguration) {
        serverConfiguration.load(readConfigurationFile(configFile));
    }

    public ServerConfiguration loadConfig() {
        return loadConfig(true, true);
    }

    public ServerConfiguration loadConfig(boolean saveDirty, boolean encryptedPasswords) {
        if (configFile.exists()) {
            ServerConfiguration serverConfiguration = readConfiguration(configFile, encryptedPasswords);
            if(!serverConfiguration.isValid()) {
                throw new Error("Configuration " + configFile + " is not valid please run with -setup option.");
            }

            if (saveDirty) {
                saveDirtyConfiguration(serverConfiguration);
            }

            return serverConfiguration;
        } else {
            throw new Error("Configuration " + configFile
                    + " does not exist, please run with -setup option before starting the server or command line interface.");
        }
    }

    static ServerConfiguration readConfiguration(File configFile, boolean encryptedPasswords) {
        ServerConfiguration serverConfiguration = new ServerConfiguration();
        serverConfiguration.setFile(configFile);
        serverConfiguration.load(readConfigurationFile(configFile), encryptedPasswords);
        return serverConfiguration;
    }

    static Properties readConfigurationFile(File configFile) {
        Properties properties = new Properties();
        try (InputStream inputStream = new FileInputStream(configFile)) {
            properties.load(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return properties;
    }

    public void writeConfiguration(ServerConfiguration serverConfiguration) {
        Properties properties = new Properties();
        serverConfiguration.save(properties);
        try (OutputStream output = new BufferedOutputStream(new FileOutputStream(configFile))) {
            String comment = "XL Deploy configuration file.\n\nNotes: \n"
                    + " * If you modify this file, you must restart the XL Deploy server.\n"
                    + " * Use `repository.keystore.password` property to specify the password used to open the repository keystore file, \n"
                    + "   which contains the encryption key for the rest of the passwords.\n"
                    + " * The other passwords specified in this file are normally encrypted, but you can change their value using plaintext \n"
                    + "   and at the next start of the server the modified passwords will be re-encrypted.\n";
            properties.store(output, comment);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void saveDirtyConfiguration(ServerConfiguration serverConfiguration) {
        if (serverConfiguration.isDirty()) {
            logger.info("Saving (dirty) configuration");
            writeConfiguration(serverConfiguration);
        }
    }

    @Override
    public String toString() {
        return configFile.getAbsolutePath();
    }

    private static final Logger logger = LoggerFactory.getLogger(ServerConfigFile.class);
}
