package com.xebialabs.ascode.yaml.writer.support

import java.util.{Collection => JavaCollection, Map => JavaMap}

import com.xebialabs.ascode.yaml.sugar.SugarConfig
import com.xebialabs.ascode.yaml.sugar.Sugarizer._
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, SourceArtifact}

class DefaultGeneratedPropertyFilter extends GeneratedPropertyFilter {
  def isInternalArtifact: PartialFunction[ConfigurationItem, Boolean] = {
    case artifact: SourceArtifact => Option(artifact.getFileUri).exists(_.startsWith("internal:"))
    case _ => false
  }

  private def getBlacklistedFields(ci: ConfigurationItem)(implicit sugarConfig: SugarConfig): Set[String] = {
    ci match {
      case artifact: Artifact if isInternalArtifact(artifact) => Set("fileUri")
      case _ => Set()
    }
  }

  override def shouldGenerate(ci: ConfigurationItem, property: PropertyDescriptor, excludedCategories: Set[String])
                             (implicit sugarConfig: SugarConfig, writerConfig: WriterConfig): Boolean = {
    val writeDefaults = writerConfig.writeDefaults
    val isBlacklisted = getBlacklistedFields(ci).contains(property.getName)
    val value = property.get(ci)
    val isEmptyValue = value match {
      case x: JavaCollection[_] => x.isEmpty
      case x: JavaMap[_, _] => x.isEmpty
      case x => x == null
    }
    val isDefaultValue = !isEmptyValue && value.equals(property.getDefaultValue)
    val isAsContainmentToParent = property.isAsContainment && property.getKind.equals(PropertyKind.CI)
    val isAnInvalidCategory = excludedCategories.contains(property.getCategory)
    val removedBySugar = ci.getType.isFieldRemovedBySugar(property.getName)
    val isWhitelisted = property.isWhitelisted(ci)

    if ((!removedBySugar && !isBlacklisted && !isEmptyValue && (writeDefaults || !isDefaultValue) && !isAsContainmentToParent && !isAnInvalidCategory) || isWhitelisted && !isEmptyValue && (writeDefaults || !isDefaultValue)) {
      val isRequiredOrHasDefault = property.isRequired || property.getDefaultValue != null

      return isRequiredOrHasDefault || (property.getKind match {
        case PropertyKind.STRING => !value.asInstanceOf[String].isEmpty
        case PropertyKind.BOOLEAN => value.asInstanceOf[Boolean]
        case PropertyKind.SET_OF_STRING | PropertyKind.LIST_OF_STRING => !value.asInstanceOf[JavaCollection[String]].isEmpty
        case PropertyKind.SET_OF_CI | PropertyKind.LIST_OF_CI => !value.asInstanceOf[JavaCollection[ConfigurationItem]].isEmpty
        case PropertyKind.MAP_STRING_STRING => !value.asInstanceOf[JavaMap[String, String]].isEmpty
        case _ => true
      })
    }
    false
  }
}

trait GeneratedPropertyFilter {
  def isInternalArtifact: PartialFunction[ConfigurationItem, Boolean]
  def shouldGenerate(ci: ConfigurationItem, property: PropertyDescriptor, excludedCategories: Set[String])
                    (implicit sugarConfig: SugarConfig, writerConfig: WriterConfig): Boolean
}
