package com.xebialabs.ascode.yaml.writer.support

import java.util
import java.util.{List => JavaList}

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.sugar.GenerateStrategy.{Filter, Reject, Skip}
import com.xebialabs.ascode.yaml.sugar.Sugarizer._
import com.xebialabs.ascode.yaml.sugar.{Add, Change, SugarConfig}
import com.xebialabs.ascode.yaml.writer.AdditionalFile
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.ascode.yaml.writer.support.descriptor.CustomPropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xltype.serialization.{CiReference, CiWriter, ConfigurationItemConverter}

import scala.collection.convert.ImplicitConversions._

class DefaultGeneratorStrategy(val propertyFilter: GeneratedPropertyFilter = new DefaultGeneratedPropertyFilter()) extends GeneratorStrategy {
  def transformCi(ci: ConfigurationItem): ConfigurationItem = ci

  override def getReferencedCiId(value: Any, references: JavaList[CiReference]): String = value match {
    case string: String => string
    case ci: ConfigurationItem => ci.getId
    case _ => throw new AsCodeException(s"Cannot infer CI type from: $value")
  }

  override val childrenResolution: Boolean = true

  override def getId(ci: ConfigurationItem, prop: PropertyDescriptor, suffix: Option[String]): String = {
    s"${ci.getId} ${prop.getName}${stringFormat(suffix)}"
  }

  override val resolveTitle: (ConfigurationItem, SugarConfig) => String = resolveCiKeyValue
}

trait GeneratorStrategy {
  val propertyFilter: GeneratedPropertyFilter
  val childrenResolution: Boolean
  val resolveTitle: (ConfigurationItem, SugarConfig) => String

  protected val stringFormat: Option[String] => String = (input: Option[String]) => input.map(str => s" $str").getOrElse("")

  def transformCi(ci: ConfigurationItem): ConfigurationItem

  def getReferencedCiId(value: Any, references: JavaList[CiReference]): String

  def getId(ci: ConfigurationItem, prop: PropertyDescriptor, suffix: Option[String] = None): String

  protected def resolveCiKeyValue(ci: ConfigurationItem, sugarConfig: SugarConfig): String = {
    ci
      .getType
      .getAllRelatedSugarDescriptors(sugarConfig)
      .flatMap(_.generateHints.idField)
      .headOption
      .flatMap(field => Option(ci.getType.getDescriptor.getPropertyDescriptor(field)).map(_.get(ci).asInstanceOf[String]))
      .getOrElse(ci.getId)
  }
}

class ConfigurationItemGenerator(strategy: GeneratorStrategy = new DefaultGeneratorStrategy(),
                                 excludeCategories: Set[String] = Set())
                                (implicit sugarConfig: SugarConfig, writerConfig: WriterConfig) extends ConfigurationItemConverter {
  private var cisToWrite = Map[String, ConfigurationItem]()
  var additionalFiles: List[AdditionalFile] = List[AdditionalFile]()
  var secrets: Map[String, String] = Map.empty

  private def isGeneratable(ci: ConfigurationItem): Boolean = {
    val sugar = ci.getType.getClosestTypeSugar

    sugar.isEmpty || sugar.forall(_.generateHints.strategy match {
      case Skip | Reject => false
      case Filter(condition) => condition(ci.getType)
    })
  }

  private def idDepth(id: String) = id.count(_ == '/')

  private def getDirectOrphans(ci: ConfigurationItem) = {
    cisToWrite
      .keySet
      .withFilter { id =>
        val parentDepth = idDepth(ci.getId)
        val childDepth = idDepth(id)

        val hasSameParent = id.split("/").dropRight(1).mkString("/").equals(ci.getId)
        val isJustOneLevelDeeperThenParent = (parentDepth + 1) == childDepth

        hasSameParent && isJustOneLevelDeeperThenParent
      }
      .flatMap(cisToWrite.get)
      .toList
  }

  override def writeCis(cis: util.Collection[ConfigurationItem], writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    cisToWrite = cisToWrite ++ cis.map(ci => ci.getId -> ci).toMap
    super.writeCis(cis, writer, ciRefsFromLevel)
  }

  private def addSugar(ci: ConfigurationItem, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    ci
      .getType
      .getClosestTypeSugar
      .foreach(_.sugarActions.foreach {
        case Add(fieldName, value) => this.writeProperty(
          ci,
          CustomPropertyDescriptor(name = Some(fieldName), value = Some(value), kind = Some(PropertyKind.STRING)),
          writer,
          ciRefsFromLevel
        )
        case Change(originalFieldName, replaceFieldName) =>
          Option(ci.getType.getDescriptor.getPropertyDescriptor(originalFieldName)).foreach(property =>
            this.writeProperty(
              ci,
              CustomPropertyDescriptor(delegate = Some(property), name = Some(replaceFieldName)),
              writer,
              ciRefsFromLevel
            )
          )
        case _ =>
      })
  }

  override def writeCi(ci: ConfigurationItem, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    if (!cisToWrite.contains(ci.getId) && strategy.childrenResolution) {
      return
    }
    cisToWrite = cisToWrite - ci.getId
    if (isGeneratable(ci)) {
      val transformedCi = strategy.transformCi(ci)
      writer.startCi(transformedCi.getType.toString, strategy.resolveTitle(ci, sugarConfig))
      if (transformedCi != ci) writeProperties(transformedCi, writer, ciRefsFromLevel)
      transformedCi match {
        case item: BaseConfigurationItem =>
          Option(item.get$token).map(_.trim).filterNot(_.isEmpty).foreach(writer.token)
          writer.ciAttributes(item.get$ciAttributes)
      }
      writeValidationMessages(ci, writer)
      writeProperties(ci, writer, ciRefsFromLevel)
      addSugar(ci, writer, ciRefsFromLevel)
      writer.endCi()
    }
  }

  private def writeFileProperty(ci: ConfigurationItem, writer: CiWriter): Unit = {
    if (strategy.propertyFilter.isInternalArtifact(ci)) {
      val artifact = ci.asInstanceOf[Artifact]
      val fileName = s"artifacts/${ci.getId}/${artifact.getFile.getName}"
      additionalFiles = AdditionalFile(fileName, artifact.getFile) :: additionalFiles
      writer.startProperty("file")
      writer.valueAsString(TaggedValue("file", fileName))
      writer.endProperty()
    }
  }

  private def writeChildrenProperty(ci: ConfigurationItem, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    val orphans: List[ConfigurationItem] = getDirectOrphans(ci).filter(isGeneratable)
    if (orphans.nonEmpty) {
      writer.startProperty("children")
      writeCis(orphans, writer, ciRefsFromLevel - 1)
      writer.endProperty()
    }
  }

  override def writeProperties(ci: ConfigurationItem, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    super.writeProperties(ci, writer, ciRefsFromLevel)
    if (strategy.childrenResolution)
      writeChildrenProperty(ci, writer, ciRefsFromLevel)
    writeFileProperty(ci, writer)
  }

  private def writePasswordProperty(ci: ConfigurationItem, property: PropertyDescriptor, writer: CiWriter): Unit = {
    Option(property.get(ci)).foreach { value =>
      property.getKind match {
        case PropertyKind.STRING =>
          writer.startProperty(property.getName)
          val (uniqueKey, updatedSecrets) = SecretsIdUtils.addToMapUnique(strategy.getId(ci, property), value.toString, secrets)
          secrets = updatedSecrets
          writer.valueAsString(TaggedValue("value", uniqueKey))
        case PropertyKind.MAP_STRING_STRING =>
          writer.startProperty(property.getName)
          val stringToString = value.asInstanceOf[util.Map[String, String]].map { case (k, v) =>
            val (uniqueKey, updatedSecrets) = SecretsIdUtils.addToMapUnique(strategy.getId(ci, property, Some(k)), v.toString, secrets)
            secrets = updatedSecrets
            (k, TaggedValue("value", uniqueKey))
          }
          writer.mapAsStrings(stringToString)
        case PropertyKind.SET_OF_STRING | PropertyKind.LIST_OF_STRING =>
          writer.startProperty(property.getName)
          val stringCollection = value.asInstanceOf[util.Collection[String]].zipWithIndex.map { case (v, i) =>
            val (uniqueKey, updatedSecrets) = SecretsIdUtils.addToMapUnique(strategy.getId(ci, property, Some(i.toString)), v.toString, secrets)
            secrets = updatedSecrets
            TaggedValue("value", uniqueKey)
          }
          writer.valuesAsStrings(stringCollection)
        case _ =>
      }
      writer.endProperty()
    }
  }

  override def writeProperty(ci: ConfigurationItem, property: PropertyDescriptor, writer: CiWriter, ciRefsFromLevel: Int): Unit = {
    if (strategy.propertyFilter.shouldGenerate(ci, property, excludeCategories)) {
      if (property.isPassword) {
        writePasswordProperty(ci, property, writer)
      } else {
        super.writeProperty(ci, property, writer, ciRefsFromLevel)
      }
    }
  }

  override def writeCiProperty(value: scala.Any, propertyDescriptor: PropertyDescriptor, writer: CiWriter): Unit = {
    writer.ciReference(strategy.getReferencedCiId(value, getReferences))
  }
}
