package com.xebialabs.ascode.yaml.sugar

import com.fasterxml.jackson.databind.node.ObjectNode
import com.fasterxml.jackson.databind.{JsonNode, ObjectMapper}
import com.xebialabs.ascode.utils.TypeSugar._

import scala.jdk.CollectionConverters._
import scala.collection.immutable.ListMap

object DeSugarProcessor {
  private def getDeSugarActions(node: JsonNode)(implicit mapper: ObjectMapper, sugarConfig: SugarConfig) = {
    if (node.has("type")) {
      val ciType = typeOf(node.get("type").asText().trim)
      sugarConfig.typeToDescriptors.get(ciType).toList.flatMap(_.deSugarActions)
    } else {
      val fields = node.fields().asScala.map(_.getKey).toSet
      sugarConfig
        .keysToDescriptors
        .find { case (field, _) => fields.contains(field) }
        .toList
        .flatMap { case (_, descriptor) => descriptor.deSugarActions }
    }
  }

  private def desugarFields(originalNode: JsonNode,
                            newNode: ObjectNode,
                            keyToDescriptor: Map[String, SugarDescriptor])
                           (implicit mapper: ObjectMapper, sugarConfig: SugarConfig): ObjectNode = {
    getDeSugarActions(originalNode).foldLeft(newNode) { (acc, currentNode) =>
      currentNode match {
        case Change(originalFieldName, replaceFieldName) =>
          Option(originalNode.get(originalFieldName)).foreach { value =>
            acc.remove(originalFieldName)
            acc.set[JsonNode](replaceFieldName, value)
          }
          newNode
        case Add(fieldName, value) =>
          acc.put(fieldName, value)
        case Remove(fieldName) =>
          acc.remove(fieldName)
          acc
      }
    }
  }

  def desugarSpec(node: JsonNode)(implicit mapper: ObjectMapper, sugarConfig: SugarConfig): JsonNode = {
    val keyToDescriptor = sugarConfig.keysToDescriptors
    val nodeFields = node.elements().asScala.map { innerNode =>
      val fields = ListMap(innerNode.fields().asScala.map(field => field.getKey -> field.getValue).toSeq: _*)
      val newNode = mapper.createObjectNode()
      fields.foreach {
        case (fieldName, fieldValue) if fieldValue.isArray && fieldValue.elements.asScala.exists(_.isObject) =>
          newNode.set[JsonNode](fieldName, desugarSpec(fieldValue))
        case (fieldName, fieldValue) =>
          newNode.set[JsonNode](fieldName, fieldValue)
      }
      desugarFields(innerNode, newNode, keyToDescriptor)
    }.toList

    nodeFields.foldLeft(mapper.createArrayNode())((emptyArray, nodeObj) => emptyArray.add(nodeObj))
  }
}
