package com.xebialabs.ascode.utils

import com.xebialabs.ascode.yaml.sugar.{Change, SugarConfig}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}

object Utils {

  implicit class IdString(id: String) {
    def parent: Option[String] = id.lastIndexOf("/") match {
      case idx if idx <= 0 => None
      case idx => Some(id.substring(0, idx))
    }

    def isRootNode: Boolean = !id.contains("/")
  }

  implicit class ExtendedList[A](list: List[A]) {
    def filterOpt(condition: Option[A => Boolean]): List[A] = condition match {
      case Some(cond) => list.filter(cond)
      case _ => list
    }

    def toOption: Option[List[A]] = list match {
      case Nil => None
      case value => Some(value)
    }
  }

  def pickFromMapByKeys[V, M >: Map[String, V]](map: M with Map[String, V], keys: Set[String]): M =
    map.filter { case (key, _) => keys.contains(key) }

  def mapMinusKeys[V, M >: Map[String, V]](map: M with Map[String, V], keys: Set[String]): M = map -- keys

  def getDuplicatedObjects[T](list: List[T]): List[T] = {
    list.groupBy(identity).collect { case (x, List(_, _, _*)) => x }.toList
  }

  def combineOptions[T](leftOption: Option[T], rightOption: Option[T], combinator: (T, T) => T): Option[T] =
    leftOption.flatMap(left => rightOption.map(right => combinator(left, right)).orElse(Some(left))).orElse(rightOption)

  def getSugaredField(propertyDescriptor: PropertyDescriptor, ciType: Type, config: SugarConfig): String = {
    val fieldName = propertyDescriptor.getName
    config
      .typeToDescriptors
      .get(ciType)
      .to(LazyList)
      .flatMap(_.sugarActions)
      .flatMap {
        case Change(`fieldName`, replaceFieldName) => Some(replaceFieldName)
        case _ => None
      }
      .headOption
      .getOrElse(fieldName)
  }

  def getName(ciPath: String): Option[String] = {
    splitStringByPathSeparator(ciPath).lastOption
  }

  def splitStringByPathSeparator(path: String): Array[String] = {
    Option(path).map(_.split("(?<!\\\\)/").dropWhile(_.isEmpty)).getOrElse(Array.empty)
  }

  def escapePath(path: String): String = {
    path.replace("\\", "\\\\").replace("/", "\\/")
  }
}
