package com.xebialabs.deployit.core.rest

import java.io.InputStream
import java.lang.annotation.Annotation
import java.lang.reflect.{ParameterizedType, Type}
import com.xebialabs.deployit.core.rest.SCMTraceabilityDataMessageBodyReader._
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData

import javax.ws.rs.core.{MediaType, MultivaluedMap}
import javax.ws.rs.ext.{MessageBodyReader, Provider}
import org.joda.time.DateTime
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import java.util.Base64
import scala.jdk.CollectionConverters._

object SCMTraceabilityDataMessageBodyReader {
  implicit class EnhancedMultivaluedMap(map: MultivaluedMap[String, String]) {
    def optionalHeader(key: String): Option[String] = {
      Option(map.get(key).asScala)
        .toList
        .flatten
        .headOption
    }

    def requiredHeader(key: String): String = optionalHeader(key).getOrElse {
      throw new Exception(s"Missing required SCM header: $key")
    }
  }
}

@Component
@Provider
class SCMTraceabilityDataMessageBodyReader extends MessageBodyReader[Option[SCMTraceabilityData]] {
  override def isReadable(`type`: Class[_],
                          genericType: Type,
                          annotations: Array[Annotation],
                          mediaType: MediaType): Boolean = {
    val isOption = classOf[Option[_]].isAssignableFrom(`type`)
    val isTraceabilityData = genericType match {
      case pt: ParameterizedType => pt.getActualTypeArguments.head == classOf[SCMTraceabilityData]
      case _ => false
    }
    isOption && isTraceabilityData
  }

  override def readFrom(`type`: Class[Option[SCMTraceabilityData]],
                        genericType: Type,
                        annotations: Array[Annotation],
                        mediaType: MediaType,
                        httpHeaders: MultivaluedMap[String, String],
                        entityStream: InputStream): Option[SCMTraceabilityData] = {
    httpHeaders.optionalHeader("X-Xebialabs-Scm-Type").map { kind =>
      new SCMTraceabilityData(
        kind,
        httpHeaders.requiredHeader("X-Xebialabs-Scm-Commit"),
        httpHeaders.requiredHeader("X-Xebialabs-Scm-Author"),
        DateTime.parse(httpHeaders.requiredHeader("X-Xebialabs-Scm-Date")),
        getDecoded64Message(httpHeaders.optionalHeader("X-Xebialabs-Scm-Message").orNull),
        httpHeaders.optionalHeader("X-Xebialabs-Scm-Remote").orNull,
        httpHeaders.requiredHeader("X-Xebialabs-Scm-Filename")
      )
    }
  }

  /**
    * Get a base 64 encoded commit message
    *
    * @return a decoded String
    */
  private def getDecoded64Message(message: String): String = try new String(Base64.getDecoder.decode(message), StandardCharsets.UTF_8)
  catch {
    case _ => message
  }
}
