package com.xebialabs.ascode.yaml.parser.util

import com.fasterxml.jackson.databind.JsonNode
import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter
import org.joda.time.DateTime

import scala.jdk.CollectionConverters._

object JsonNodeSugar {

  private implicit class ExtendedOption[T](option: Option[T]) {
    def reportFieldIsMissing(field: String): Option[T] = option
      .orElse(throw new AsCodeException(s"Document is missing a [$field] field in spec."))
  }

  implicit class JsonNodeAccessors(node: JsonNode) {
    def property(name: String): Option[JsonNode] = Option(node.get(name))

    def requiredProperty(name: String): JsonNode = Option(node.get(name)).reportFieldIsMissing(name).get

    def string(propertyName: String): Option[String] = property(propertyName).map(_.asText(""))

    def boolean(propertyName: String): Option[Boolean] = property(propertyName).map(_.booleanValue())

    def requiredString(propertyName: String): String = string(propertyName)
      .reportFieldIsMissing(propertyName)
      .get

    def requiredBoolean(propertyName: String): Boolean = boolean(propertyName)
      .reportFieldIsMissing(propertyName)
      .get

    def dateTime(propertyName: String): Option[DateTime] = string(propertyName).map(new DateTimeAdapter().unmarshal)

    def listOfStrings(propertyName: String): List[String] =
      property(propertyName).toList.flatMap(_.iterator().asScala.map(_.asText("")))

    def mapStringString(propertyName: String): Map[String, String] =
      property(propertyName).toList.flatMap(objProperty =>
        objProperty.fieldNames().asScala.map(field => field -> objProperty.get(field).asText(""))
      ).toMap

    private def convertToMapStringAnyRef(node: Option[JsonNode]): Map[String, AnyRef] = node
      .toList
      .flatMap(_.fields().asScala.map { field =>
        field.getKey -> (field.getValue match {
          case double if double.isNumber => java.lang.Double.valueOf(double.asDouble())
          case list if list.isArray => list.asScala.toList.map(list => list.asText("")).asJava
          case keyValueMap if keyValueMap.isObject => keyValueMap.fields().asScala.toList.map(field => field.getKey -> field.getValue.asText("")).toMap.asJava
          case value => value.asText("")
        })
      }).toMap

    def mapStringAnyRef(propertyName: String): Map[String, AnyRef] = convertToMapStringAnyRef(property(propertyName))

    def requiredMapStringAnyRef(propertyName: String): Map[String, AnyRef] =
      convertToMapStringAnyRef(property(propertyName).reportFieldIsMissing(propertyName))
  }

}
