package com.xebialabs.deployit.core.rest

import java.io.Writer

import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.core.io.IOContext
import com.fasterxml.jackson.core.{JsonParser, ObjectCodec}
import com.fasterxml.jackson.dataformat.yaml.{YAMLFactory, YAMLGenerator, YAMLMapper}
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.deployit.core.rest.YamlSupport.CustomYAMLFactory
import org.yaml.snakeyaml.DumperOptions.{ScalarStyle, Version}
import org.yaml.snakeyaml.events.{Event, ImplicitTuple, ScalarEvent}

object YamlSupport {
  private val booleanLiterals = Set("n", "y", "1", "0", "yes", "no")

  class CustomYAMLGenerator(ctxt: IOContext,
                            jsonFeatures: Int,
                            yamlFeatures: Int,
                            codec: ObjectCodec,
                            out: Writer,
                            version: Version) extends YAMLGenerator(ctxt, jsonFeatures, yamlFeatures, codec, out, version) {
    override def writeString(text: String): Unit = {
      if (booleanLiterals.contains(text.toLowerCase)) {
        _verifyValueWrite("write String value")
        _writeScalar(text, "string", '"')
      }
      else super.writeString(text)
    }

    def emitCustomEvent(event: Event): Unit = _emitter.emit(event)

    def contextWriteValue(): Int = {
      _writeContext.writeValue
    }

    def contextWriteFieldName(name: String): Int = {
      _writeContext.writeFieldName(name)
    }
  }

  class CustomYAMLFactory extends YAMLFactory {
    override def _createGenerator(out: Writer, ctxt: IOContext): YAMLGenerator =
      new CustomYAMLGenerator(ctxt, _generatorFeatures, _yamlGeneratorFeatures, _objectCodec, out, _version)
  }

  implicit class EnhancedYAMLGenerator(yamlGenerator: YAMLGenerator) {
    def emitEvent(event: Event): Unit = {
      yamlGenerator match {
        case custom: CustomYAMLGenerator => custom.emitCustomEvent(event)
      }
    }

    def contextWriteValue(): Int = {
      yamlGenerator match {
        case custom: CustomYAMLGenerator => custom.contextWriteValue()
      }
    }

    def writeTaggedString(tag: String, value: String): Unit = {
      yamlGenerator.contextWriteValue()
      yamlGenerator
        .emitEvent(new ScalarEvent(
          null,
          s"!$tag",
          new ImplicitTuple(false, false),
          value,
          null,
          null,
          ScalarStyle.DOUBLE_QUOTED
        ))
    }

    def writeTaggedStringField(field: String, tag: String, value: String): Unit = {
      yamlGenerator.writeFieldName(field)
      writeTaggedString(tag, value)
    }
  }
}

trait YamlSupport {
  implicit val yamlFactory: YAMLFactory = new CustomYAMLFactory()
  yamlFactory.enable(JsonParser.Feature.STRICT_DUPLICATE_DETECTION)
  yamlFactory.enable(YAMLGenerator.Feature.MINIMIZE_QUOTES)
  yamlFactory.enable(YAMLGenerator.Feature.ALWAYS_QUOTE_NUMBERS_AS_STRINGS)

  implicit val yamlMapper: YAMLMapper = new YAMLMapper(yamlFactory)
  yamlMapper.registerModule(DefaultScalaModule)
  yamlMapper.setSerializationInclusion(Include.NON_EMPTY)
}
