package com.xebialabs.ascode.yaml.parser

import java.io.InputStream

import com.fasterxml.jackson.core.JsonParseException
import com.fasterxml.jackson.databind.node.ObjectNode
import com.fasterxml.jackson.dataformat.yaml.YAMLParser
import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.Specs
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.deployit.core.rest.YamlSupport

import scala.collection.convert.ImplicitConversions._
import scala.util.{Failure, Success, Try}

object DefinitionParser {
  def specs(product: String, specParsers: (List[String], SpecParser)*): Specs[SpecParser] =
    Specs(specParsers.flatMap { case (kinds, parser) =>
      kinds.map(kind => (product -> kind) -> parser)
    }.toMap)

  def apply(product: String, specParsers: (List[String], SpecParser)*): DefinitionParser = {
    new DefinitionParser(specs(product, specParsers: _*))
  }
}

class DefinitionParser(specs: Specs[SpecParser]) extends YamlSupport {
  def validateMetadata(metadata: Map[String, String]): Unit = {
    val validHomes = specs.specs.keySet.map { case (_, kind) => s"$kind-home" } + "home"
    metadata.keys.foreach(metadataHome => {
      if (!validHomes.exists(metadataHome.equals)) {
        AsCodeException.throwDocumentFieldException("metadata", s"Invalid home defined: $metadataHome")
      }
    })
  }

  private def parseDefinition(node: ObjectNode) = {
    if (!node.has("apiVersion") || node.get("apiVersion").isNull) {
      AsCodeException.throwDocumentFieldException("apiVersion", "Document is missing an apiVersion definition")
    }
    val apiVersion = node.get("apiVersion").asText()
    if (apiVersion.isEmpty || apiVersion.indexOf('/') == -1) {
      AsCodeException.throwDocumentFieldException(
        "apiVersion",
        s"apiVersion format invalid: ${Option(apiVersion).getOrElse("")}"
      )
    }
    val project = apiVersion.split('/').head

    if (!node.has("kind") || node.get("kind").isNull) {
      AsCodeException.throwDocumentFieldException("kind", "Document is missing a kind definition")
    }
    val kind = node.get("kind").asText()

    val specParser = specs.getSpec(project, kind)

    val metadata = getMetadata(node)
    metadata.foreach(validateMetadata)

    if (!node.has("spec") || node.get("spec").isNull) {
      AsCodeException.throwDocumentFieldException("spec", "Document is missing a spec definition")
    }

    Definition(
      apiVersion,
      metadata,
      kind,
      specParser.parse(kind, metadata, node.get("spec"))
    )
  }

  private def getMetadata(node: ObjectNode) =
    Option(node.get("metadata"))
      .map(_.fields.map(field => field.getKey -> field.getValue.asText(null)).toMap)

  def getParseExceptionMessage(e: Throwable): String = e match {
    case ex: JsonParseException => s"${ex.getOriginalMessage} at line number: ${ex.getLocation.getLineNr}"
    case ex if ex.getCause != null => getParseExceptionMessage(ex.getCause)
    case _@ex => ex.getMessage
  }

  private def parseYaml(yamlParser: YAMLParser) =
    Try(yamlParser.readValuesAs(classOf[ObjectNode]).toList) match {
      case Success(document :: Nil) => parseDefinition(document)
      case Success(_) => throw new AsCodeException("This endpoint handles only one document per request.")
      case Failure(e) => throw new AsCodeException(getParseExceptionMessage(e))
    }

  def parse(stream: InputStream): Definition = parseYaml(yamlFactory.createParser(stream))

  def parse(string: String): Definition = parseYaml(yamlFactory.createParser(string))
}