package com.xebialabs.ascode.yaml.dto

import com.xebialabs.ascode.utils.Utils._
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.{Changes, Errors, TaskInfo}

object AsCodeResponse {
  object EntityKinds {
    implicit class ExtendedString(string: String) {
      def ids: ChangedIds = ChangedIds(string)
    }

    val CI = "CI"
    val PERMISSION = "PERMISSION"
    val ROLE = "ROLE"
    val USER = "USER"
  }

  case class ChangedIds(kind: String, created: List[String] = Nil, updated: List[String] = Nil) {
    def merge(changed: ChangedIds): ChangedIds = ChangedIds(
      kind,
      created = created ::: changed.created,
      updated = updated ::: changed.updated
    )

    def withCreated(ci: String): ChangedIds = this.copy(created = created :+ ci)

    def withCreated(cis: List[String]): ChangedIds = this.copy(created = this.created ::: cis)

    def withCreated(cis: String*): ChangedIds = this.withCreated(cis.toList)

    def withUpdated(ci: String): ChangedIds = this.copy(updated = updated :+ ci)

    def withUpdated(cis: List[String]): ChangedIds = this.copy(updated = this.updated ::: cis)

    def withUpdated(cis: String*): ChangedIds = this.withUpdated(cis.toList)

    def toOption: Option[ChangedIds] = {
      if (created.nonEmpty || updated.nonEmpty) {
        Some(this)
      } else None
    }
  }

  case class TaskInfo(id: String, description: String, started: Boolean)

  case class Changes(ids: List[ChangedIds] = Nil, task: Option[TaskInfo] = None) {
    def merge(other: Changes): Changes = {
      val task = this.task.orElse(other.task)
      val changedIds = (ids ::: other.ids)
        .groupBy(_.kind)
        .mapValues(_.reduce(_.merge(_)))
        .values
        .toList
        .sortBy(_.kind)

      Changes(changedIds, task)
    }
  }

  case class CiValidationError(ciId: String, propertyName: String, message: String)

  case class PermissionError(ciId: String, permission: String)

  case class DocumentFieldError(field: String, problem: String)

  case class Errors(validation: List[CiValidationError] = Nil,
                    permission: List[PermissionError] = Nil,
                    document: Option[DocumentFieldError] = None,
                    generic: Option[String] = None) {
    def merge(other: Errors): Errors = {
      Errors(
        validation = this.validation ::: other.validation,
        permission = this.permission ::: other.permission,
        document = this.document.orElse(other.document),
        generic = this.generic.orElse(other.generic)
      )
    }
  }

  def errors(errors: Errors) = AsCodeResponse(errors = Some(errors))

  def changes(changes: Changes) = AsCodeResponse(changes = Some(changes))

  def genericError(error: String): AsCodeResponse = errors(Errors(generic = Some(error)))

  def ids(changedIds: ChangedIds*): AsCodeResponse = changes(Changes(ids = changedIds.toList))

  def cis(created: List[String] = Nil, updated: List[String] = Nil): AsCodeResponse =
    ids(CI.ids.withCreated(created).withUpdated(updated))

  def task(task: TaskInfo): AsCodeResponse = changes(Changes(task = Some(task)))
}

case class AsCodeResponse(changes: Option[Changes] = None, errors: Option[Errors] = None) {
  def withTask(id: String, description: String, started: Boolean): AsCodeResponse = this.copy(
    changes = this.changes.orElse(Some(Changes())).map(_.copy(task = Some(TaskInfo(id, description, started))))
  )

  def merge(other: AsCodeResponse): AsCodeResponse = {
    AsCodeResponse(
      changes = combineOptions[Changes](this.changes, other.changes, _.merge(_)),
      errors = combineOptions[Errors](this.errors, other.errors, _.merge(_))
    )
  }
}