[CmdletBinding()]
param()

Trace-VstsEnteringInvocation $MyInvocation

try
{
	[string]$connectedServiceName = Get-VstsInput -Name connectedServiceName -Require
	[string]$manifestPath = Get-VstsInput -Name manifestPath -Require
	[bool]$publish = Get-VstsInput -Name publish -AsBool
	[bool]$version = Get-VstsInput -Name version -AsBool
	[bool]$deploy = Get-VstsInput -Name deploy -AsBool
	[bool]$rollback = Get-VstsInput -Name rollback -AsBool
	[string]$targetEnvironment = Get-VstsInput -Name targetEnvironment
    [string]$outputFolder = Get-VstsInput -Name outputFolder
    [string]$rootFolder = Get-VstsInput -Name rootFolder
    [string]$versionNumber = Get-VstsInput -Name versionNumber

	Import-Module -Name $PSScriptRoot\ps_modules\xld-import.psm1
	Import-Module -Name $PSScriptRoot\ps_modules\xld-package.psm1
	Import-Module -Name $PSScriptRoot\ps_modules\xld-verify.psm1
	Import-Module -Name $PSScriptRoot\ps_modules\xld-deploy.psm1
	
	# Check if path to manifest file is valid
	Assert-VstsPath $manifestPath -PathType Leaf
	
	# Verify output folder
	if ($outputFolder)
	{
		# if folder doesn't exist or it's not valid
		if (-not (Test-Path $outputFolder -PathType Container -IsValid))
		{
			throw "Provided output folder is not a valid path."
		}
	}
	else
	{
		$outputFolder = $env:BUILD_STAGINGDIRECTORY
	}

	# Verify root folder
	if ($rootFolder)
	{
		# if folder doesn't exist or it's not valid
		if (-not (Test-Path $rootFolder -PathType Container -IsValid))
		{
			throw "Provided root folder is not a valid path."
		}
	}
	else
	{
		$rootFolder = $env:BUILD_STAGINGDIRECTORY
	}

	$serviceEndpoint = Get-EndpointData $connectedServiceName

	# Add URL and credentials to default parameters so that we don't need
	# to specify them over and over for this session.
	$PSDefaultParameterValues.Add("*:EndpointUrl", $serviceEndpoint.Url)
	$PSDefaultParameterValues.Add("*:Credential", $serviceEndpoint.Credential)

	# Check server state and validate the address
	Write-Output "Checking XL Deploy server state..."
	if ((Get-ServerState) -ne "RUNNING")
	{
		throw "XL Deploy server not reachable. Address or credentials are invalid or server is not in a running state."
	}
	Write-Output "XL Deploy server is running and credentials are validated."

	# Substitute version place holder with version number in manifest file
	if ($version)
	{		
		if (-not $versionNumber)
		{
			$versionNumber = $env:BUILD_BUILDNUMBER
		}

		Write-Verbose ("Setting manifest version to {0}..." -f $versionNumber)	

		$null = Set-Version $manifestPath $versionNumber
		
		Write-Verbose "Package version set to $versionNumber."
	}

	# Create DAR package
	Write-Output "Creating DAR package..."
	$packageFullPath = New-DarPackage $manifestPath $outputFolder $rootFolder
	Write-Output "DAR package created at $packageFullPath."

	# Upload package to XL Deploy
	Write-Output "Starting package upload..."
	$deploymentPackageId = Send-PackageEx $packageFullPath
	Write-Output "Package successfully uploaded. Package ID is $deploymentPackageId"

	# Publish Build artifacts to the TFS server
	if ($publish)
	{
		$folderName = "XL Deploy - Deployment Archives"
        $packageInfo = Get-PackageInfo $packageFullPath
        $artifactName = $packageInfo.Application + " " + $packageInfo.Version

		Write-Output "Publishing build artifact $artifactName..."

		Write-VstsUploadArtifact -ContainerFolder $folderName -Name $artifactName -Path $packageFullPath
		Write-Output "Build artifact published successfully."
	}

	# Trigger the deployment if target environment is specified
	if ($deploy)
	{
		Write-Output "Starting deployment to $targetEnvironment."
		# create new deployment task
		$deploymentTaskId = New-DeploymentTask $deploymentPackageId $targetEnvironment

		Start-Task $deploymentTaskId

		$taskOutcome = Get-TaskOutcome $deploymentTaskId

		if ($taskOutcome -eq "EXECUTED" -or $taskOutcome -eq "DONE")
		{
			# archive
			Complete-Task $deploymentTaskId
			Write-Output "Successfully deployed to $targetEnvironment."
		}
		else
		{
			if (!$rollback) 
			{
				throw "Deployment failed."
			}

			Write-Warning "Deployment failed."
			Write-SetResult "SucceededWithIssues" "Deployment failed."
			
			Write-Output "Starting rollback."

			# rollback
			$rollbackTaskId = New-RollbackTask $deploymentTaskId

			Start-Task $rollbackTaskId

			$rollbackTaskOutcome = Get-TaskOutcome $rollbackTaskId

			if ($rollbackTaskOutcome -eq "EXECUTED" -or $rollbackTaskOutcome -eq "DONE")
			{
				# archive
				Complete-Task $rollbackTaskId
				Write-Output "Rollback executed successfully."
			}
			else
			{
				throw "Rollback failed." 
			}
		}
	}
}
finally
{
    Trace-VstsLeavingInvocation $MyInvocation
}