package com.xebialabs.deployit.cli.api;

import static java.lang.String.format;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.api.internal.ImportHelper;
import com.xebialabs.deployit.cli.api.internal.PrintHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.dto.Comparison;
import com.xebialabs.deployit.core.api.dto.ConfigurationItem;
import com.xebialabs.deployit.core.api.dto.ImportablePackages;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.RepositoryObjects;
import com.xebialabs.deployit.core.api.dto.ServerInfo;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.dto.TaskInfo;

/**
 * I am the main API class, exposing calls into the deep innards of the Deployit tool.
 */
@CliObject(name = "deployit")
@ClassHelp(description = "The main gateway to interfacing with Deployit.")
public class DeployitClient extends DocumentedObject {
    private final Proxies proxies;

    // Needed for tests

    public DeployitClient() {
        this.proxies = null;
    }

    public DeployitClient(Proxies proxies)  {
        this.proxies = proxies;
    }

    @MethodHelp(description = "Import a package located on the server or local file system.", parameters = {
        @ParameterHelp(name = "importablePackage", description = "This is either:\n\t- The name of the importable package on the server\n\t- The absolute path to a local importable package.")
    })
    public RepositoryObject importPackage(final String importablePackageName) {
        return ImportHelper.doImport(proxies.getImportablePackage(), importablePackageName);
    }

    public List<String> listImportablePackages() {
        return ((ImportablePackages) new ResponseExtractor(proxies.getImportablePackage().list()).getEntity()).getFiles();
    }

    @MethodHelp(description = "Deploy or upgrade a deployment-package on the targeted environment using the mappings specified, and wait for it to be completed.", parameters = {
        @ParameterHelp(name = "source", description = "The source deployment-package that is to be deployed"),
        @ParameterHelp(name = "target", description = "Can either be:\n\t- The target environment to which you want to deploy\n\t- The existing deployment that needs to be upgraded"),
        @ParameterHelp(name = "mappings", description = "The mappings to use for the deployment, if no mappings are specified, default mappings will be used.")
    }, returns = "The id of the task that was executed")
    public String deployAndWait(String source, String target, RepositoryObject... mappings) {
        final String taskId = prepareDeployment(source, target, mappings);
        startTaskAndWait(taskId);
        return taskId;
    }

    @MethodHelp(description = "Deploy or upgrade a deployment-package on the targeted environment using default mappings, and wait for it to be completed.", parameters = {
        @ParameterHelp(name = "source", description = "The source deployment-package that is to be deployed"),
        @ParameterHelp(name = "target", description = "Can either be:\n\t- The target environment to which you want to deploy\n\t- The existing deployment that needs to be upgraded")
    }, returns = "The id of the task that was executed")
    public String deployAndWait(String source, String target) {
        final String taskId = prepareDeployment(source, target);
        startTaskAndWait(taskId);
        return taskId;
    }

    public void startTaskAndWait(final String taskId) {
        startTask(taskId);
        // Wait until done/failed
         boolean done = false;
         TaskInfo ti = null;
         while (!done) {
             ti = new ResponseExtractor(proxies.getTaskRegistry().getTaskInfo(taskId)).getEntity();
             final String st = ti.getState();
             if ("DONE".equals(st) || "ABORTED".equals(st) || "STOPPED".equals(st)) {
                 done = true;
             }
             try {
                 Thread.sleep(1000);
             } catch (InterruptedException e) {
                 Thread.currentThread().interrupt();
             }
         }
       }

    @MethodHelp(description = "Deploy or upgrade a deployment-package on the targeted environment using the mappings specified and return immediately.", parameters = {
        @ParameterHelp(name = "source", description = "The source deployment-package that is to be deployed"),
        @ParameterHelp(name = "target", description = "Can either be:\n\t- The target environment to which you want to deploy\n\t- The existing deployment that needs to be upgraded"),
        @ParameterHelp(name = "mappings", description = "The mappings to use for the deployment, if no mappings are specified, default mappings will be used.")
    }, returns = "The id of the task that was started")
    public String deploy(final String source, final String target, final RepositoryObject[] mappings) {
        final String taskid = prepareDeployment(source, target, mappings);
        startTask(taskid);
        return taskid;
    }

    @MethodHelp(description = "Deploy or upgrade a deployment-package on the targeted environment using default mappings and return immediately.", parameters = {
        @ParameterHelp(name = "source", description = "The source deployment-package that is to be deployed"),
        @ParameterHelp(name = "target", description = "Can either be:\n\t- The target environment to which you want to deploy\n\t- The existing deployment that needs to be upgraded")
    }, returns = "The id of the task that was started")
    public String deploy(final String source, final String target) {
        final String taskid = prepareDeployment(source, target);
        startTask(taskid);
        return taskid;
    }
    
    @MethodHelp(description = "Generate all default mappings for deployment/upgrade", parameters = {
            @ParameterHelp(name = "source", description = "The list of a package member ids"),
            @ParameterHelp(name = "target", description = "The target environment id to which you want to deploy")
    }, returns = "The ids of the generated default mappings")
    public RepositoryObject[] generateAllMappings(String source, String target) {
        RepositoryObjects mappingsDto = new ResponseExtractor(proxies.getDeployment().generateMappings(source, target)).getEntity();
        final List<RepositoryObject> mappings = mappingsDto.getObjects();
        return mappings.toArray(new RepositoryObject[mappings.size()]);
    }
    
    @MethodHelp(description = "Generate default mappings.", parameters = {
            @ParameterHelp(name = "sources", description = "The list of a package member ids"),
            @ParameterHelp(name = "target", description = "The target environment id to which you want to deploy")
    }, returns = "The ids of the generated default mappings")
    public RepositoryObject[] generateMappings(List<String> sources, String target) {
        RepositoryObjects mappings = new ResponseExtractor(proxies.getDeployment().generatePartialMappings(sources, target, null)).getEntity();
        return mappings.getObjects().toArray(new RepositoryObject[mappings.getObjects().size()]);
    }
    
    @MethodHelp(description = "Generate default mappings.", parameters = {
            @ParameterHelp(name = "sources", description = "The list of a package member ids"),
            @ParameterHelp(name = "target", description = "The target environment member id to which you want to deploy"),
            @ParameterHelp(name = "mappingType", description = "The mapping type to be used for creating the default mappings")
    }, returns = "The ids of the generated default mappings")
    public RepositoryObject[] generateMappings(List<String> sources, String target, String mappingType) {
        RepositoryObjects mappings = new ResponseExtractor(proxies.getDeployment().generatePartialMappings(sources, target, mappingType)).getEntity();
        return mappings.getObjects().toArray(new RepositoryObject[mappings.getObjects().size()]);
    }

    @MethodHelp(description = "Prepare Deployment or upgrade of a deployment-package on the targeted environment using default mappings and returns the taskid.", parameters = {
        @ParameterHelp(name = "source", description = "The source deployment-package that is to be deployed"),
        @ParameterHelp(name = "target", description = "Can either be:\n\t- The target environment to which you want to deploy\n\t- The existing deployment that needs to be upgraded")
    }, returns = "The id of the task that was prepared")
    public String prepareDeployment(final String source, final String target) {
        RepositoryObjects mappingsDto = (RepositoryObjects) new ResponseExtractor(proxies.getDeployment().generateMappings(source, target)).getEntity();
        final List<RepositoryObject> mappings = mappingsDto.getObjects();
        return prepareDeployment(source, target, mappings.toArray(new RepositoryObject[mappings.size()]));
    }

    @MethodHelp(description = "Prepare a (un)deployment or upgrade of a deployment-package on the targeted environment using the mappings specified and returns the taskid.", parameters = {
        @ParameterHelp(name = "source", description = "The source deployment-package that is to be deployed"),
        @ParameterHelp(name = "target", description = "Can either be:\n\t- The target environment to which you want to deploy\n\t- The existing deployment that needs to be upgraded"),
        @ParameterHelp(name = "mappings", description = "The mappings to use for the deployment")
    }, returns = "The id of the task that was prepared")
    public String prepareDeployment(final String source, final String target, final RepositoryObject[] mappings) {
        RepositoryObjects mappingsDto = new RepositoryObjects();
        if (mappings != null && mappings.length > 0) {
            mappingsDto.setObjects(Arrays.asList(mappings));
        }

        // validate the mappings
	    final ResponseExtractor responseExtractor = new ResponseExtractor(proxies.getDeployment().validate(source, target, mappingsDto));
	    if (responseExtractor.isValidResponse()) {
			// prepare the deployment
			Steps steps = new ResponseExtractor(proxies.getDeployment().prepare(source, target, mappingsDto)).getEntity();
			return steps.getTaskId();
        } else {
		    final RepositoryObjects validated = responseExtractor.getEntity();
		    for (RepositoryObject v : validated.getObjects()) {
			    if (!v.getValidations().isEmpty()) {
				    System.out.println(format("mapping with id %s has the following validation errors %s", v.getId(), v.getValidations()));
			    }
		    }
		    return null;
        }
    }

    @MethodHelp(description = "Retrieve the task information for a (running) deployment task.", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    }, returns = "The metadata associated with the task")
    public TaskInfo retrieveTaskInfo(final String taskId) {
        return new ResponseExtractor(proxies.getTaskRegistry().getTaskInfo(taskId)).getEntity();
    }

    @MethodHelp(description = "Gracefully stop an active deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void stopTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().stop(taskId));
    }

    @MethodHelp(description = "Start a deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void startTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().start(taskId));
    }

    @MethodHelp(description = "Abort an active deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void abortTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().abort(taskId));
    }

    @MethodHelp(description = "Cancel a stopped deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void cancelTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().cancel(taskId));
    }
    
    @MethodHelp(description = "(Un)skip steps of the task.\n- If a step is in the PENDING or FAILED state, it will be SKIPPED.\n- If a step is in the SKIPPED state it will be PENDING. ", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task"),
        @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip or unskip")
    })
    public void skipSteps(String taskId, Integer[] stepIds) {
        final String ids = StringUtils.join(stepIds, ",");
        new ResponseExtractor(proxies.getTaskRegistry().toggleSkipSteps(taskId, ids));
    }

    @MethodHelp(description = "Do a reality check on the middleware specified by the id. The updated values are returned and should be manually stored.", parameters = {
        @ParameterHelp(name = "id", description = "The id of the discoverable middleware component")
    }, returns = "The updated (but not persisted) repository object.")
    public RepositoryObject realityCheck(String id) {
        return new ResponseExtractor(proxies.getDiscovery().realityCheck(id)).getEntity();
    }

    @MethodHelp(description = "Discovers middleware associated with the specified discoverable configuration item. Refer to plugin configuration item documentation to see which fields are required for discover.", parameters = {
        @ParameterHelp(name = "ci", description = "The configuration item with all required discovery fields filled in")
    }, returns = "RepositoryObjects containing all discovered middleware associated with the specified discoverable ci")
    public RepositoryObjects discover(ConfigurationItem ci) {
        return new ResponseExtractor(proxies.getDiscovery().discover(ci)).getEntity();
    }

    @MethodHelp(description = "Describe the CI class, with all the values it takes.", parameters = {
        @ParameterHelp(name = "shortName", description = "The (Short) name of the CI eg. \"Host\", \"WasDataSource\", etc")
    })
    public void describe(String typeName) {
        DescriptorHelper.describe(typeName);
    }

    @MethodHelp(description = "Print a tree-view of a CI", parameters = {
        @ParameterHelp(name = "ci", description = "The CI to print")
    })
    public void print(RepositoryObject ci) {
        PrintHelper.printCi(ci, proxies);
    }

    public Comparison compare(String reference, List<String> ids) {
        return new ResponseExtractor(proxies.getQuery().compare(reference, ids)).getEntity();
    }

	public ServerInfo info() {
		return new ResponseExtractor(proxies.getServer().getInfo()).getEntity();
	}
}
