package com.xebialabs.deployit.cli.api;

import java.util.Collection;
import java.util.List;

import com.google.common.base.Joiner;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.api.internal.ImportHelper;
import com.xebialabs.deployit.cli.api.internal.PrintHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.dto.Comparison;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDtos;
import com.xebialabs.deployit.core.api.dto.ImportablePackages;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.ServerInfo;
import com.xebialabs.deployit.core.api.dto.TaskInfo;
import com.xebialabs.deployit.core.api.dto.TaskInfos;


/**
 * I am the main API class, exposing calls into the deep innards of the Deployit tool.
 */
@CliObject(name = "deployit")
@ClassHelp(description = "The main gateway to interfacing with Deployit.")
public class DeployitClient extends DocumentedObject {
    private final Proxies proxies;

    // Needed for tests
    public DeployitClient() {
        this.proxies = null;
    }

    public DeployitClient(Proxies proxies)  {
        this.proxies = proxies;
    }

    @MethodHelp(description = "Import a package located on the server or local file system.", parameters = {
        @ParameterHelp(name = "importablePackage", description = "This is either:\n\t- The name of the importable package on the server\n\t- The absolute path to a local importable package.")
    })
    public RepositoryObject importPackage(final String importablePackageName) {
        return new ImportHelper().doImport(proxies.getAuthentication().getHttpClient(), proxies.getHost(), importablePackageName);
    }

    @MethodHelp(description = "List all importable packages on the server.")
    public List<String> listImportablePackages() {
        return ((ImportablePackages) new ResponseExtractor(proxies.getImportablePackage().list()).getEntity()).getFiles();
    }

    @MethodHelp(description = "Retrieve the task information for a (running) deployment task.", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    }, returns = "The metadata associated with the task")
    public TaskInfo retrieveTaskInfo(final String taskId) {
        return new ResponseExtractor(proxies.getTaskRegistry().getTaskInfo(taskId)).getEntity();
    }

    @MethodHelp(description = "Gracefully stop an active deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void stopTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().stop(taskId));
    }

    @MethodHelp(description = "Start a deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void startTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().start(taskId));
    }

	@MethodHelp(description = "Start a deployment task and wait for it to finish", parameters = {
	    @ParameterHelp(name = "taskId", description = "The id of the task.")
	})
	public void startTaskAndWait(final String taskId) {
	     startTask(taskId);
	     // Wait until done/failed
	      boolean done = false;
	      TaskInfo ti = null;
	      while (!done) {
	          ti = new ResponseExtractor(proxies.getTaskRegistry().getTaskInfo(taskId)).getEntity();
	          final String st = ti.getState();
	          if ("DONE".equals(st) || "ABORTED".equals(st) || "STOPPED".equals(st)) {
	              done = true;
	          }
	          try {
	              Thread.sleep(1000);
	          } catch (InterruptedException e) {
	              Thread.currentThread().interrupt();
	          }
	      }
	}

    @MethodHelp(description = "Abort an active deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void abortTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().abort(taskId));
    }

    @MethodHelp(description = "Cancel a stopped deployment task", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void cancelTask(String taskId) {
        new ResponseExtractor(proxies.getTaskRegistry().cancel(taskId));
    }
    
    @MethodHelp(description = "List unfinished tasks for the current user.")
    public List<TaskInfo> listUnfinishedTasks() {
        return new ResponseExtractor(proxies.getTaskRegistry().getUnfinishedTasks()).<TaskInfos>getEntity().getTasks();
    }
    
    @MethodHelp(description = "List all unfinished tasks registered in the server. Requires administrative permission.")
    public List<TaskInfo> listAllUnfinishedTasks() {
        return new ResponseExtractor(proxies.getTaskRegistry().getAllUnfinishedTasks()).<TaskInfos>getEntity().getTasks();
    }
    
    @MethodHelp(description = "(Un)skip steps of the task.\n- If a step is in the PENDING or FAILED state, it will be SKIPPED.\n- If a step is in the SKIPPED state it will be PENDING. ", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task"),
        @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip or unskip")
    })
    public void skipSteps(String taskId, Integer[] stepIds) {
        final String ids = Joiner.on(",").join(stepIds);
        new ResponseExtractor(proxies.getTaskRegistry().toggleSkipSteps(taskId, ids));
    }

    @MethodHelp(description = "Assign a pending task to another principal", parameters = {
        @ParameterHelp(name = "taskId", description = "The id of the task."),
        @ParameterHelp(name = "owner", description = "The new task owner.")
    })
    public void assignTask(String taskId, String owner) {
    	new ResponseExtractor(proxies.getTaskRegistry().assignTask(taskId, owner));
    }

    @MethodHelp(description = "Discover middleware associated with the specified discoverable configuration item. Refer to plugin configuration item documentation to see which fields are required for discover.", parameters = {
        @ParameterHelp(name = "ci", description = "The configuration item with all required discovery fields filled in")
    }, returns = "RepositoryObjects containing all discovered middleware associated with the specified discoverable ci")
    public ConfigurationItemDtos discover(ConfigurationItemDto ci) {
        return new ResponseExtractor(proxies.getDiscovery().discover(ci)).getEntity();
    }

    @MethodHelp(description = "Execute the control task on the provided configuration item. If the control tasks executes successfully, no output is shown. In the case of an error, log output is shown.", parameters = {
            @ParameterHelp(name = "taskName", description = "The control task name eg. \"start\""),
            @ParameterHelp(name = "ci", description = "The configuration item to execute the control task on")
        })
	public void executeControlTask(String taskName, ConfigurationItemDto ci) {
		new ResponseExtractor(proxies.getControlTask().execute(taskName, ci.getId()));
	}

    @MethodHelp(description = "Describe the CI class, with all the values it takes.", parameters = {
        @ParameterHelp(name = "shortName", description = "The (Short) name of the CI eg. \"Host\", \"WasDataSource\", etc")
    })
    public void describe(String typeName) {
        DescriptorHelper.describe(typeName);
    }

    @MethodHelp(description = "Print a tree-view of a CI", parameters = {
        @ParameterHelp(name = "ci", description = "The CI to print")
    })
    public void print(RepositoryObject ci) {
    	PrintHelper.getInstance().printCi(ci, proxies);      
    }

	public void print(Collection<RepositoryObject> cis) {
		PrintHelper.getInstance().printCis(cis, proxies);
	}

    public Comparison compare(String reference, List<String> ids) {
        return new ResponseExtractor(proxies.getQuery().compare(reference, ids)).getEntity();
    }

	@MethodHelp(description = "Retrieve Deployit server information")
	public ServerInfo info() {
		return new ResponseExtractor(proxies.getServer().getInfo()).getEntity();
	}

	@MethodHelp(description = "Run the Deployit server garbage collector")
	public void runGarbageCollector() {
		new ResponseExtractor(proxies.getServer().gc());
	}
	
	@MethodHelp(description = "Shutdown the Deployit server")
	public void shutdown() {
		new ResponseExtractor(proxies.getServer().shutdown());
	}
}
