package com.xebialabs.deployit.cli.api.internal;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.engine.api.RepositoryService;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.google.common.collect.Collections2.transform;

public class PrintHelper {

    private final Set<String> repositoryObjectParentIds = Sets.<String> newHashSet();

    private PrintHelper() {
    }

    public final static PrintHelper getInstance() {
        return new PrintHelper();
    }

    public void printCi(final ConfigurationItem object, final RepositoryService repository) {
        printTopLevelCiWithIndent(object, "", repository);
    }

    public void printCis(Collection<ConfigurationItem> cis, RepositoryService repository) {
        repositoryObjectParentIds.addAll(transform(cis, new Function<ConfigurationItem, String>() {
            @Override
            public String apply(ConfigurationItem input) {
                return input.getId();
            }
        }));
        for (ConfigurationItem ci : cis) {
            printTopLevelCiWithIndent(ci, "", repository);
        }
    }

    private void printTopLevelCiWithIndent(final ConfigurationItem object, final String indent, final RepositoryService repository) {
        System.out.println(object.getType());
        printCiProperties(object, indent, repository);
    }

    private void printCiProperties(final ConfigurationItem object, final String indent, final RepositoryService repository) {
        repositoryObjectParentIds.add(object.getId());
        println(indent, "id", object.getId(), false);
        println(indent, "values", null, true);
        printValues(object, (indent + "    "), repository);
    }

    @SuppressWarnings("unchecked")
    private void printValues(final ConfigurationItem object, final String indent, final RepositoryService repository) {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(object.getType());
        Iterator<PropertyDescriptor> iterator = descriptor.getPropertyDescriptors().iterator();
        while (iterator.hasNext()) {
            PropertyDescriptor pd = iterator.next();
            Object value = pd.get(object);

            if (value == null)
                continue;

            boolean last = !iterator.hasNext();
            String deepIndent = indent + (iterator.hasNext() ? "|   " : "    ");
            switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
            case ENUM:
                println(indent, pd, value.toString(), last);
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                println(indent, pd, null, last);
                final Collection<String> strings = (Collection<String>) value;
                for (Iterator<String> stringIt = strings.iterator(); stringIt.hasNext();) {
                    String string = stringIt.next();
                    println(deepIndent, string, null, !stringIt.hasNext());
                }
                break;
            case MAP_STRING_STRING:
                println(indent, pd, null, last);
                final Map<String, String> stringMap = (Map<String, String>) value;
                for (Map.Entry<String, String> mapEntry : stringMap.entrySet()) {
                    println(deepIndent, mapEntry.getKey(), mapEntry.getValue(), !iterator.hasNext());
                }
                break;
            case CI:
                final String id = (String) value;
                if (!this.repositoryObjectParentIds.contains(id)) {
                    final ConfigurationItem nested = repository.read(id);
                    println(indent, pd, nested.getType().toString(), last);
                    printCiProperties(nested, deepIndent, repository);
                } else {
                    println(indent, pd, "REFERENCE => " + id, last);
                }
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                println(indent, pd, null, last);
                Collection<String> ids = (Collection<String>) value;
                printSetOfCis(deepIndent, ids, repository);
                break;
            default:
                throw new IllegalArgumentException("Unknown PropertyKind: " + pd.getKind() + " for " + pd.getFqn());
            }
        }
    }

    private void printSetOfCis(final String indent, final Collection<String> ids, final RepositoryService repository) {
        for (Iterator<String> idIt = ids.iterator(); idIt.hasNext();) {
            String nestedid = idIt.next();
            if (!this.repositoryObjectParentIds.contains(nestedid)) {
                final ConfigurationItem nestedCi = repository.read(nestedid);
                println(indent, nestedCi.getType().toString(), null, !idIt.hasNext());
                printCiProperties(nestedCi, indent + (idIt.hasNext() ? "|   " : "    "), repository);
            } else {
                println(indent, "REFERENCE => " + nestedid, null, !idIt.hasNext());
            }
        }
    }

    private static String getIndicator(PropertyDescriptor pd) {
        if (pd.isHidden()) {
            return " (hidden)";
        }
        return "";
    }

    private static void println(String indent, String key, String value, boolean last) {
        System.out.println(indent + (last ? "\\-- " : "+-- ") + key + (value != null ? ": " + value : ""));
    }

    private static void println(String indent, PropertyDescriptor pd, String value, boolean last) {
        System.out.println(indent + (last ? "\\-- " : "+-- ") + pd.getName() + (value != null ? ": " + value : "") + getIndicator(pd));
    }
}
