package com.xebialabs.deployit.cli;

import com.xebialabs.deployit.cli.help.HelpScanner;
import jline.console.ConsoleReader;
import jline.console.completer.Completer;

import static java.lang.System.out;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static jline.internal.Preconditions.checkNotNull;

public class CliCompleter implements Completer {

    private ConsoleReader consoleReader;

    public CliCompleter(ConsoleReader consoleReader) {
        this.consoleReader = consoleReader;
    }

    private List<CharSequence> filterStringsByBuffer(List<String> strings, String buffer) {
        List<CharSequence> candidates = new ArrayList<>();
        if (buffer == null) {
            candidates.addAll(strings);
        } else {
            for (String match : strings) {
                if (match.startsWith(buffer)) {
                    candidates.add(match);
                }
            }
        }
        return candidates;
    }

    public int complete(String buffer, int cursor, final List<CharSequence> candidates) {
        checkNotNull(candidates);
        if (buffer == null) {
            return -1;
        }

        String phrase = buffer.substring(0, cursor);
        String[] parts = phrase.split("\\.");

        boolean completeCliObjects = parts.length == 1 && !buffer.endsWith(".");
        boolean completeCliObjectMethods = parts.length == 2 || (parts.length == 1 && buffer.endsWith("."));
        boolean completeCliObjectMethodSignature = parts.length == 2 && buffer.endsWith("(");

        List<String> completionStrings = new ArrayList<>();
        if (completeCliObjects) {
            completionStrings.addAll(HelpScanner.loadCliObjectCompletions());
            completionStrings.addAll(Arrays.asList("help", "quit", "exit"));
            List<CharSequence> filteredStrings = filterStringsByBuffer(completionStrings, buffer);
            candidates.addAll(filteredStrings);
        } else if (completeCliObjectMethods) {
            String cliObjectName = parts[0];
            if (completeCliObjectMethodSignature) {
                String cliMethodName = parts[1].substring(0, parts[1].length() - 1);
                List<String> helpSuggestions = HelpScanner.loadMethodSignatureSuggestions(cliObjectName, cliMethodName);
                printPossibleMethodSignatures(buffer, cursor, helpSuggestions);
            } else {
                completionStrings = HelpScanner.loadObjectMethodCompletions(cliObjectName);
                if (!completionStrings.isEmpty()) {
                    completionStrings.add(cliObjectName + ".help()");
                }
            }
            List<CharSequence> filteredStrings = filterStringsByBuffer(completionStrings, buffer);
            candidates.addAll(filteredStrings);

            addOpeningBracketIfMethodFound(candidates);
        }
        return candidates.isEmpty() ? -1 : 0;
    }

    private void printPossibleMethodSignatures(String buffer, int cursor, List<String> helpSuggestions) {
        out.printf("\n");
        for (String help : helpSuggestions) {
            out.printf(help);
        }
        try {
            consoleReader.resetPromptLine(consoleReader.getPrompt(), buffer, cursor);
        } catch (IOException e) {
            //eat away the exception, as we don't want to show errors in auto-completion
        }
    }

    private void addOpeningBracketIfMethodFound(List<CharSequence> candidates) {
        if (candidates.size() == 1) {
            String method = candidates.get(0).toString();
            if (!method.endsWith(")")) {
                String methodWithOpenParenthesis = candidates.get(0) + "(";
                candidates.remove(0);
                candidates.add(methodWithOpenParenthesis);
            }
        }
    }
}