package com.xebialabs.deployit.cli.api;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Calendar;
import java.util.List;

import com.google.common.base.Strings;
import com.xebialabs.deployit.cli.rest.RequestExecutor;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock;

import org.apache.commons.lang.CharEncoding;
import org.apache.http.client.methods.HttpPost;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Function;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.ExportHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.core.api.dto.Revision;
import com.xebialabs.deployit.engine.api.dto.ArtifactAndData;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.cli.api.ObjectFactory.toType;
import static com.xebialabs.deployit.cli.help.DateHelp.*;
import static java.lang.String.format;

@CliObject(name = "repository")
@ClassHelp(description = "Gateway to doing CRUD operations on all types of CIs")
public class RepositoryClient extends DocumentedObject {
    public static final Function<ConfigurationItemId, String> ciIdToString = new Function<ConfigurationItemId, String>() {
        public String apply(ConfigurationItemId input) {
            return input.getId();
        }
    };
    private ProxiesInstance proxies;
    private DeployitCommunicator communicator;
    private RequestExecutor requestExecutor;

    public RepositoryClient() {}

    public RepositoryClient(final ProxiesInstance proxies) {
        this.proxies = proxies;
        this.communicator = proxies.getCommunicator();
        this.requestExecutor = new RequestExecutor(proxies.getCommunicator());
    }

    @MethodHelp(description = "Create a new CI in the repository", parameters = {
            @ParameterHelp(name = "ci", description = "The CI (ConfigurationItem) that should be created in the repository") })
    public ConfigurationItem create(final ConfigurationItem object) {
        checkArgument(object.getId() != null, "The repository object should have a set id.");
        ConfigurationItem created = proxies.getRepository().create(object.getId(), object);
        return checkForValidations(created);
    }

    @MethodHelp(description = "Create a new artifact CI in the repository", parameters = {
            @ParameterHelp(name = "artifact", description = "The Artifact that should be created in the repository") })
    public ConfigurationItem create(final com.xebialabs.deployit.engine.api.dto.ArtifactAndData artifact) {
        checkArgument(artifact.getArtifact().getId() != null, "The artifact should have a set id.");
        ConfigurationItem configurationItem = proxies.getRepository().create(artifact.getArtifact().getId(), artifact);
        return checkForValidations(configurationItem);
    }

    @MethodHelp(description = "Create all new CIs in the repository, commonly used after a discovery", parameters = {
            @ParameterHelp(name = "cis", description = "The CIs (ConfigurationItems) that should be created in the repository") })
    public List<ConfigurationItem> create(final List<ConfigurationItem> cis) {
        final List<ConfigurationItem> response = proxies.getRepository().create(cis);
        return checkAllForValidations(response);
    }

    private static List<ConfigurationItem> checkAllForValidations(List<ConfigurationItem> cis) {
        for (ConfigurationItem ci : cis) {
            if (ci instanceof ValidatedConfigurationItem) {
                ValidatedConfigurationItem vci = (ValidatedConfigurationItem) ci;
                if (!vci.getValidations().isEmpty()) {
                    logger.error("Configuration item contained validation errors: {}", vci.getValidations());
                }
            }
        }
        return cis;
    }

    @MethodHelp(description = "Read a CI form the repository", parameters = { @ParameterHelp(name = "id", description = "The id of the CI to read") })
    public ConfigurationItem read(final String id) {
        return proxies.getRepository().read(id);
    }

    @MethodHelp(description = "Update an existing CI in the repository", parameters = {
            @ParameterHelp(name = "ci", description = "The updated CI (ConfigurationItem) that should be stored in the repository") })
    public ConfigurationItem update(final ConfigurationItem object) {
        checkArgument(object.getId() != null, "The repository object should have a set id.");
        return checkForValidations(proxies.getRepository().update(object.getId(), object));
    }

    @MethodHelp(description = "Update an existing artifact in the repository", parameters = {
            @ParameterHelp(name = "artifact", description = "The updated artifact CI that should be stored in the repository") })
    public ConfigurationItem update(final ArtifactAndData artifact) {
        checkArgument(artifact.getArtifact().getId() != null, "The repository object should have a set id.");
        return checkForValidations(proxies.getRepository().update(artifact.getArtifact().getId(), artifact));
    }

    @MethodHelp(description = "Creates or updates all CIs in the repository, commonly used after a discovery", parameters = {
            @ParameterHelp(name = "cis", description = "The CIs (ConfigurationItems) that should be created or updated in the repository") })
    public List<ConfigurationItem> update(final List<ConfigurationItem> cis) {
        final List<ConfigurationItem> response = proxies.getRepository().update(cis);
        return checkAllForValidations(response);
    }

    @MethodHelp(description = "Move a CI from one location to another", parameters = {
            @ParameterHelp(name = "ci", description = "The CI that is to be moved"),
            @ParameterHelp(name = "newId", description = "The new id of the CI")
    }, returns = "The CI with the new id")
    public ConfigurationItem move(final ConfigurationItem ci, final String newId) {
        return move(ci.getId(), newId);
    }

    @MethodHelp(description = "Move a CI from one location to another", parameters = {
            @ParameterHelp(name = "id", description = "The id of the CI that is to be moved"),
            @ParameterHelp(name = "newId", description = "The new id of the CI")
    }, returns = "The CI with the new id")
    public ConfigurationItem move(final String id, final String newId) {
        return proxies.getRepository().move(id, newId);
    }

    @MethodHelp(description = "Copy a configuration item in the repository. The item as well as all its children are copied to the new location. " +
            "The parent reflected in the new location, must be of the same type as that of the parent of the item being copied.", parameters = {
            @ParameterHelp(name = "id", description = "The id of the CI that is to be copied"),
            @ParameterHelp(name = "newId", description = "The new id of the CI")
    }, returns = "The CI with the new id")
    public ConfigurationItem copy(final String id, final String newId) {
        return proxies.getRepository().copy(id, newId);
    }

    @MethodHelp(description = "Rename a CI", parameters = {
            @ParameterHelp(name = "ci", description = "The CI to rename"),
            @ParameterHelp(name = "newName", description = "The new name (last part of the id)")
    }, returns = "The CI with the updated name")
    public ConfigurationItem rename(final ConfigurationItem ci, final String newName) {
        return rename(ci.getId(), newName);
    }

    @MethodHelp(description = "Rename a CI", parameters = {
            @ParameterHelp(name = "id", description = "The id of the CI to rename"),
            @ParameterHelp(name = "newName", description = "The new name (last part of the id)")
    }, returns = "The CI with the updated name")
    public ConfigurationItem rename(String id, String newName) {
        return proxies.getRepository().rename(id, newName);
    }

    @MethodHelp(description = "Delete a CI with a specific id from the repository", parameters = { @ParameterHelp(name = "id", description = "The id of the CI") })
    public void delete(final String id) {
        proxies.getRepository().delete(id);
    }

    @MethodHelp(description = "Delete a CIs with a specific list of ids from the repository", parameters = { @ParameterHelp(name = "ids", description = "The list of ids of the CI") })
    public void deleteList(final List<String> ids) {
        proxies.getRepository().deleteList(ids);
    }

    @MethodHelp(description = "Search for CIs of a specific type in the repository", parameters = { @ParameterHelp(name = "ciType", description = "") }, returns = "The ids of the configuration items that fit the query.")
    public List<String> search(String ciType) {
        Type t = null;
        if (ciType != null) {
            if (!DescriptorRegistry.exists(toType(ciType))) {
                System.err.println("Configuration item type [" + ciType + "] not known.");
                return null;
            }
            t = toType(ciType);
        }
        List<ConfigurationItemId> ids = proxies.getRepository().query(t, null, null, null, null, null, 0, -1);
        return newArrayList(Lists.transform(ids, ciIdToString));
    }

    @MethodHelp(description = "Search for CIs of a specific type in the repository which are located under the specified parent node.", parameters = {
            @ParameterHelp(name = "ciType", description = "The type of configuration item to look for (eg. udm.DeployedApplication)"),
            @ParameterHelp(name = "parent", description = "The id of the parent node to search under (eg. Environments/env1)") },
            returns = "The ids of the configuration items that fit the query.")
    public List<String> search(String ciType, String parent) {
        if (ciType != null) {
            if (!DescriptorRegistry.exists(toType(ciType))) {
                System.err.println("Configuration item type [" + ciType + "] not known.");
                return null;
            }
        }
        List<ConfigurationItemId> ids = proxies.getRepository().query(toType(ciType), parent, null, null, null, null, 0, -1);
        return newArrayList(Lists.transform(ids, ciIdToString));
    }

    @MethodHelp(description = "Search for CIs with a (partial) name in the repository.", parameters = {
            @ParameterHelp(name = "name", description = "The name of the configuration item to search for. Can contain '%' as wildcard") },
            returns = "The ids of the configuration items that match the name.")
    public List<String> searchByName(String name) {
        if (name == null) {
            System.err.println("No name specified to search for.");
            return null;
        }
        List<ConfigurationItemId> ids = proxies.getRepository().query(null, null, null, name, null, null, 0, -1);
        return newArrayList(Lists.transform(ids, ciIdToString));
    }

    @MethodHelp(description = "Search for CIs of a specific type in the repository were created before the given date.", parameters = {
            @ParameterHelp(name = "ciType", description = "The type of configuration item to look for (eg. udm.DeployedApplication)"),
            @ParameterHelp(name = "before", description = "a java.util.Calendar which specifies a moment in time before which the CI has to be created") },
            returns = "The ids of the configuration items that fit the query.")
    public List<String> search(String ciType, Calendar c) {
        List<ConfigurationItemId> ids = proxies.getRepository().query(toType(ciType), null, null, null, new DateTime(c), null, 0, -1);
        return newArrayList(Lists.transform(ids, ciIdToString));
    }

    @MethodHelp(description = "Read multiple objects from the repostory in one go.", parameters = { @ParameterHelp(name = "ids", description = "The ids of the objects to read") })
    public List<ConfigurationItem> read(List<String> ids) {
        final List<ConfigurationItem> objs = proxies.getRepository().read(ids);
        return objs;
    }

    @MethodHelp(description = "Get all task information, including steps and blocks, from the repository's archive.",
            returns = "A list of all archived tasks.")
    public List<TaskWithBlock> getArchivedTaskList() {
        return proxies.getTaskBlockRegistry().export(null, null);
    }

    @MethodHelp(description = "Get all task information, including steps, from the repository's archive in the specified date range.", parameters = {
            @ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format"),
            @ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format") },
            returns = "This object contains all archived tasks with their enclosed steps")
    public List<TaskWithBlock> getArchivedTasksList(final String beginDate, final String endDate) {
        return proxies.getTaskBlockRegistry().export(toLocalDate(beginDate), toLocalDate(endDate));
    }

    @MethodHelp(description = "Export all task information, including steps, from the repository's archive to a local XML file", parameters = {
            @ParameterHelp(name = "filePath", description = "Fully qualified pathname, including the file name, as to where to store the file to. Example: '/tmp/exportedTasks.xml'") })
    public void exportArchivedTasks(final String filePath) throws IOException {
        getResourceToWrite(filePath, "/task/export");
    }

    @MethodHelp(description = "Export all task information, including steps, from the repository's archive in the specified date range to a local XML file", parameters = {
            @ParameterHelp(name = "filePath", description = "Fully qualified pathname, including the file name, as to where to store the file to. Example: '/tmp/exportedTasks.xml'"),
            @ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format"),
            @ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format") })
    public void exportArchivedTasks(final String filePath, final String beginDate, final String endDate) throws IOException {
        getResourceToWrite(filePath, format("/task/export?begindate=%s&enddate=%s",
                urlEncode(transform(beginDate, DATE_FORMAT_IN_CLI, DATE_FORMAT_FOR_SERVER)),
                urlEncode(transform(endDate, DATE_FORMAT_IN_CLI, DATE_FORMAT_FOR_SERVER))));
    }

    @MethodHelp(description = "Export a deployment package in DAR format to a local file", parameters = {
            @ParameterHelp(name = "directoryPath", description = "Fully qualified path of the directory where the deployment package will be exported with name '<application-name>-<version>.dar'. Directory will be created if it doesn't exist"),
            @ParameterHelp(name = "versionId", description = "Id of the deployment package that needs to be exported") })
    public void exportDar(final String directoryPath, final String versionId) throws IOException {
        ConfigurationItem ci = read(versionId);
        if (!ci.getType().equals(Type.valueOf("udm", "DeploymentPackage"))) {
            System.err.println("Export expected Id corresponding to type udm.DeploymentPackage but was actually of type " + ci.getType());
        }

        File directory = new File(directoryPath);
        if (!directory.exists()) {
            directory.mkdirs();
        }

        String version = ci.getId().substring(ci.getId().lastIndexOf("/") + 1);
        String idWithoutVersion = ci.getId().substring(0, ci.getId().lastIndexOf("/"));
        String appName = idWithoutVersion.substring(idWithoutVersion.lastIndexOf("/") + 1);
        String filePath = directoryPath + File.separator + appName + "-" + version + ".dar";

        String s = proxies.getExportProxy().exportDar(ci.getId());

        getResourceToWrite(filePath, "/internal/download/" + s);
    }

    @MethodHelp(description = "Check for existence of an id, does not guarantee read access to said configuration item.", parameters = {
            @ParameterHelp(name = "id", description = "The id of the configuration item")
    }, returns = "true if there is a configuration item with that id.")
    public boolean exists(String id) {
        return proxies.getRepository().exists(id);
    }


    @MethodHelp(description = "Create a task to import CI tree from exported archive. Such archive can be created with 'repository.import'. The task is NOT started automatically.",
            parameters = {
                    @ParameterHelp(name = "archiveLocation", description = "Location of the archive") },
            returns = "Task ID which you can start using task management tools like 'task2' or 'deployit'")
    public String importCis(String archiveLocation) throws UnsupportedEncodingException {
        String uri = communicator.getConfig().getExtensionApiUrl() + "/import/citree?from=" + URLEncoder.encode(archiveLocation, CharEncoding.UTF_8);
        return requestExecutor.execute(new HttpPost(uri));
    }

    @MethodHelp(description = "Import CI tree from exported archive. Such archive can be created with 'repository.import'. The task is NOT started automatically.",
            parameters = {
                    @ParameterHelp(name = "archiveLocation", description = "Location of the archive") })
    public void importCisAndWait(String archiveLocation) throws UnsupportedEncodingException {
        String taskId = importCis(archiveLocation);
        DeployitClient.startTaskAndWait(taskId, proxies, true);
        TaskWithBlock task = proxies.getTaskBlockRegistry().getTask(taskId);
        if (!task.getState().equals(TaskExecutionState.DONE)) {
            String msg = "Something went wrong and file could not be imported. Please check server logs.";
            logger.error(msg);
            System.err.println(msg);
        }
    }

    @MethodHelp(description = "Create a task to export CI tree into default directory on the server. The task is NOT started automatically.",
            parameters = {
                    @ParameterHelp(name = "exportRootId", description = "ID of the CI to start with") },
            returns = "ID of the task which you can start using task management tools like 'task2' or 'deployit'")
    public String exportCis(String exportRootId) throws UnsupportedEncodingException {
        return exportCis(exportRootId, null);
    }

    @MethodHelp(description = "Create a task to export CI tree into specified directory on the server. The task is NOT started automatically.",
            parameters = {
                    @ParameterHelp(name = "exportRootId", description = "ID of the CI to start with"),
                    @ParameterHelp(name = "exportDir", description = "Directory on the server to export into") },
            returns = "ID of the task which you can start using task management tools like 'task2' or 'deployit'")
    public String exportCis(String exportRootId, String exportDir) throws UnsupportedEncodingException {
        return requestExecutor.execute(prepareExportPost(exportRootId, exportDir));
    }

    @MethodHelp(description = "Export CI tree into default directory on the server.",
            parameters = {
                    @ParameterHelp(name = "exportRootId", description = "ID of the CI to start with") },
            returns = "File path of the generated ZIP file, relative to the XL Deploy server location")
    public String exportCisAndWait(String exportRootId) throws UnsupportedEncodingException {
        return exportCisAndWait(exportRootId, null);
    }

    @MethodHelp(description = "Export CI tree into default directory on the server.",
            parameters = {
                    @ParameterHelp(name = "exportRootId", description = "ID of the CI to start with"),
                    @ParameterHelp(name = "exportDir", description = "Directory on the server to export into") },
            returns = "File path of the generated ZIP file, relative to the XL Deploy server location")
    public String exportCisAndWait(String exportRootId, String exportDir) throws UnsupportedEncodingException {
        String taskId = exportCis(exportRootId, exportDir);
        TaskWithBlock task = proxies.getTaskBlockRegistry().getTask(taskId);
        String exportedFile = task.getMetadata().get("exportedFile");
        if (exportedFile == null) {
            String msg = "Something went wrong and exported file name could not be found. Please check server logs.";
            logger.error(msg);
            System.err.println(msg);
        }
        DeployitClient.startTaskAndWait(taskId, proxies, true);
        return exportedFile;
    }

    private HttpPost prepareExportPost(String id, String exportDir) throws UnsupportedEncodingException {

        String uri = proxies.getCommunicator().getConfig().getExtensionApiUrl() + "/export/citree/" +
                URLEncoder.encode(Strings.nullToEmpty(id), CharEncoding.UTF_8);
        if (exportDir != null) {
            uri += "?exportDir=" + URLEncoder.encode(exportDir, CharEncoding.UTF_8);
        }
        return new HttpPost(uri);
    }


    public List<Revision> getVersionHistory(String id) {
        return proxies.getHistoryService().readRevisions(id);
    }

    private static ConfigurationItem checkForValidations(final ConfigurationItem ci) {
        if (ci instanceof ValidatedConfigurationItem && !((ValidatedConfigurationItem) ci).getValidations().isEmpty()) {
            logger.error("Configuration item contained validation errors: {}", ((ValidatedConfigurationItem) ci).getValidations());
            System.out.printf("Configuration item %s contained validation errors: %s\n", ci.getId(), ((ValidatedConfigurationItem) ci).getValidations());
        }
        return ci;
    }

    private void getResourceToWrite(final String filePath, final String resourcePath) throws IOException {
        if (new ExportHelp(communicator).writeResourceToLocalFile(filePath, communicator.getConfig().getContext() + resourcePath) > 0) {
            System.out.println(format("finished writing file to %s", filePath));
        } else {
            System.out.println("File was created but no bytes were written to the file.\n");
            System.out.println("Maybe the requested resource was not available or zero size!?");
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(RepositoryClient.class);
}
