package com.xebialabs.deployit.rest.test.api.datasets

import com.xebialabs.deployit.rest.test.api.DataSet
import com.google.common.io.Files
import com.google.common.base.Charsets
import java.io.File
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.deployit.plugin.api.udm.base.{ BaseDeployableFileArtifact, BaseDeployableFolderArtifact }
import java.util.{ Calendar, GregorianCalendar, UUID }
import com.xebialabs.deployit.plugin.api.udm.Container
import scala.collection.immutable.List
import com.xebialabs.deployit.task.TaskForReporting
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.deployit.exception.RuntimeIOException
import com.xebialabs.deployit.plugin.api.udm._

object FillRepository extends DataSet {

  def name() = "fillRepository"

  def generateSet() {
    create { c =>
      createEnvironments(c)
      createPackages(c)
    }

    createUsers
    generateTasksForReports
    importPackages

    val startDate1 = new GregorianCalendar(2010, 10, 10, 9, 30, 10)
    val startDate2 = new GregorianCalendar(2010, 10, 15, 7, 20, 56)

    val completionDate1 = new GregorianCalendar(2010, 10, 10, 10, 30, 45)
    val completionDate2 = new GregorianCalendar(2010, 10, 15, 10, 30, 45)

    deploy("Applications/tinyApp/1.0", "Environments/0tinyEnv0")
    deploy("Applications/tinyApp/1.0", "Environments/1newEnv1")
    deploy("Applications/smallApp/1.0", "Environments/0tinyEnv0")
    deploy("Applications/smallApp/1.0", "Environments/1newEnv1")
   
    task { t =>
      t.success.forApp("tinyApp").forVersion("1.0").forEnv("0tinyEnv0").startedAt(startDate1).completedAt(completionDate1).withId(UUID.randomUUID().toString())
    }

    task { t =>
      t.success.forApp("tinyApp").forVersion("1.0").forEnv("1newEnv1").startedAt(startDate1).completedAt(completionDate1).withId(UUID.randomUUID().toString())
    }

    task { t =>
      t.success.forApp("smallApp").forVersion("1.0").forEnv("0tinyEnv0").startedAt(startDate2).completedAt(completionDate2).withId(UUID.randomUUID().toString())
    }

    task { t =>
      t.success.forApp("smallApp").forVersion("1.0").forEnv("1newEnv1").startedAt(startDate2).completedAt(completionDate2).withId(UUID.randomUUID().toString())
    }
  }

  private def createEnvironments(c: Creator) {
    createEnvsWithServersAndHost(c, nrEnvs = 10, nrServers = 1, "host", kbLog = 5, nrSteps = 1)
    createEnvsWithServers(c, nrEnvs = 10, nrServers = 1, "tiny", kbLog = 5, nrSteps = 1)
    createEnvsWithServers(c, nrEnvs = 25, nrServers = 2, "small", kbLog = 3, nrSteps = 5)
    createEnvsWithServers(c, nrEnvs = 25, nrServers = 20, "large", kbLog = 2, nrSteps = 20)
    createEnvsWithServers(c, nrEnvs = 10, nrServers = 1, "new", kbLog = 5, nrSteps = 1)

    def createEnvsWithServers(c: Creator, nrEnvs: Int, nrServers: Int, name: String, kbLog: Int, nrSteps: Int) {
      createEnvs(c, name, nrEnvs, createServers(c, nrServers, name, kbLog, nrSteps))
    }

    def createEnvsWithServersAndHost(c: Creator, nrEnvs: Int, nrServers: Int, name: String, kbLog: Int, nrSteps: Int) {
      val servers = createServers(c, nrServers, name, kbLog, nrSteps)
      val host = List(c.host("Host-with-" + name + "Server") { h =>
        h.setOs("Unix")
        h.setAddress("localhost")
        h.setUsername("admin")
        h.setPassword("admin")
        h.setAccessMethod("FTP")
      })
      createEnvs(c, name, nrEnvs, servers.union(host))
    }

    def createServers(c: Creator, nrServers: Int, name: String, kbLog: Int, nrSteps: Int): List[_ <: Container] = {
      (for (i <- 0 until nrServers) yield c.server(name + "Server" + i) { s =>
        s.setAmountOfKBLogFiles(kbLog)
        s.setNumberOfSteps(nrSteps)
        s.setStepDelayTimeInMilliSeconds(1000)
      }).toList
    }

    def createEnvs(c: Creator, name: String, nrEnvs: Int, containers: List[_ <: Container]) {
      for (i <- 0 until nrEnvs) yield c.environment(i + name + "Env" + i) {
        containers
      }
    }
  }

  private def createPackages(c: Creator) {
    val tempDir = Files.createTempDir()

    createPackageWithDummyEar(c, "newApp", nrPackages = 10, packageSize = 1, "newEar")
    createPackageWithDummyEar(c, "oldApp", nrPackages = 10, packageSize = 3, "oldEar")
    createPackageWithDummyEar(c, "otherApp", nrPackages = 10, packageSize = 5, "otherEar")
    createPackageWithDummyEarAndConfigurationFile(c, "tinyApp", nrPackages = 10, packageSize = 1, "tinyEar")
    createPackageWithDummyEar(c, "smallApp", nrPackages = 100, packageSize = 2, "smallEar")
    createPackageWithDummyEar(c, "largeApp", nrPackages = 100, packageSize = 20, "largeEar")
    createPackageWithDummyEarWithProperties(c, "dictionaryApp", nrPackages = 10, packageSize = 2, "dictionaryEar")

    def createPackageWithDummyEar(c: Creator, appName: String, nrPackages: Int, packageSize: Int, earName: String) {
      createPackage(c, appName, nrPackages) { dp =>
        for (j <- 0 until packageSize) {
          createDummyEarFile(c, earName + j, dp)
          ::
        }
        Nil
      }
    }

    def createPackageWithDummyEarWithProperties(c: Creator, appName: String, nrPackages: Int, packageSize: Int, earName: String) {
      createPackage(c, appName, nrPackages) { dp =>
        for (j <- 0 until packageSize) {
          createDummyEarWithPropertiesFile(c, earName + j, dp)
          ::
        }
        Nil
      }
    }

    def createPackageWithDummyEarAndConfigurationFile(c: Creator, appName: String, nrPackages: Int, packageSize: Int, earName: String) {
      createPackage(c, appName, nrPackages) { dp =>
        val configName = appName + "config";
        for (j <- 0 until packageSize) {
          createDummyEarFile(c, earName + j, dp)
          ::
          createConfigFile(c, configName + j, dp)
          ::
        }
        Nil
      }
    }

    def createPackage(c: Creator, appName: String, nrPackages: Int)(deployables: DeploymentPackage => List[_ <: Deployable]) {
      val app = c.application(appName)
      for (i <- 0 until nrPackages) {
        c.deploymentPackage(i + ".0", app) { dp =>
          deployables.apply(dp)
        }
      }
    }

    def createDummyEarFile(c: Creator, earName: String, dp: DeploymentPackage): Deployable = {
      c.ear(earName, dp) { e =>
        e.setFile(createFile(tempDir, earName + ".ear"))
      }
    }

    def createDummyEarWithPropertiesFile(c: Creator, earName: String, dp: DeploymentPackage): Deployable = {
      c.earWithProperties(earName, dp) { e =>
        e.setFile(createFile(tempDir, earName + ".ear"))
      }
    }

    def createConfigFile(c: Creator, configName: String, dp: DeploymentPackage) = {
      c.configurationFile(configName, dp) { c =>
        c.setFile(LocalFile.valueOf({
          val directory = new File(tempDir, configName)
          if (directory.mkdir()) {
            createFile(directory, configName + ".config")
          }
          directory
        }))
      }
    }

    def createFile(tempDir: File, fileName: String) = {
      LocalFile.valueOf({
        val file: File = new File(tempDir, fileName)
        file.createNewFile()
        Files.write("**********", file, Charsets.UTF_8)
        file
      })
    }
  }

  private def createUsers {
    user("mary")
    val grantMary = grant("mary") _
    grantMary("login", "")
    grantMary("read", "Applications/smallApp")
    grantMary("read", "Applications/largeApp")
    grantMary("deploy#initial", "")

    user("john")
    val grantJohn = grant("john") _
    grantJohn("login", "")
    grantJohn("read", "Applications/tinyApp")
    grantJohn("deploy#initial", "")
  }

  private def importPackages {
    importPackage("AnimalZoo-ear/1.0")
    importPackage("AnimalZoo-ear/2.0")
    importPackage("AnimalZoo-ear/3.0")
    importPackage("AnimalZoo-ear/4.0")
    importPackage("SampleDarWithConfig/1.0")
  }

  object TaskState extends Enumeration {
    type TaskState = Value
    val SUCCESSFUL, FAILED, ABORTED = Value
  }

  import TaskState._

  private def generateTasksForReports = {
    createCurrentTask("newApp", "1.0", "0tinyEnv0", days = 30, durationInMinutes = 3, SUCCESSFUL)
    createCurrentTask("dictionaryApp", "1.0", "0tinyEnv0", days = 30, durationInMinutes = 11, ABORTED)
    createCurrentTask("otherApp", "1.0", "0smallEnv0", days = 30, durationInMinutes = 11, FAILED)
    createTask("oldApp", "1.0", "0tinyEnv0", month = 2, durationInMinutes = 7, SUCCESSFUL)
    createTask("otherApp", "1.0", "0tinyEnv0", month = 3, durationInMinutes = 11, ABORTED)
    createTask("tinyApp", "1.0", "0smallEnv0", month = 4, durationInMinutes = 4, FAILED)
    createTask("dictionaryApp", "1.0", "0smallEnv0", month = 4, durationInMinutes = 3, SUCCESSFUL)
    createTask("newApp", "1.0", "0smallEnv0", month = 5, durationInMinutes = 6, ABORTED)
    createTask("oldApp", "1.0", "0smallEnv0", month = 6, durationInMinutes = 17, FAILED)
    createTask("newApp", "1.0", "0largeEnv0", month = 6, durationInMinutes = 14, SUCCESSFUL)

    def createTask(appName: String, version: String, envName: String, month: Int, durationInMinutes: Int, state: TaskState) {
      val startDate = new GregorianCalendar()
      startDate.add(Calendar.MONTH, -month)

      val completionDate = new GregorianCalendar()
      completionDate.add(Calendar.MONTH, -month)
      completionDate.add(Calendar.MINUTE, durationInMinutes)

      task { t =>
        (state match {
          case SUCCESSFUL => t.success
          case FAILED => t.successWithFails
          case ABORTED => t.failed
        }).forApp(appName).forVersion(version).forEnv(envName).startedAt(startDate).completedAt(completionDate).withId(UUID.randomUUID().toString())

      }
    }
    
    def createCurrentTask(appName: String, version: String, envName: String, days: Int, durationInMinutes: Int, state: TaskState) {
      val startDate = new GregorianCalendar()
      startDate.add(Calendar.DATE, -days)

      val completionDate = new GregorianCalendar()
      completionDate.add(Calendar.DATE, -days)
      completionDate.add(Calendar.MINUTE, durationInMinutes)

      task { t =>
        (state match {
          case SUCCESSFUL => t.success
          case FAILED => t.successWithFails
          case ABORTED => t.failed
        }).forApp(appName).forVersion(version).forEnv(envName).startedAt(startDate).completedAt(completionDate).withId(UUID.randomUUID().toString())

      }
    }
  }

}