package com.xebialabs.deployit.cli.api;

import java.util.List;
import java.util.stream.Collectors;

import org.joda.time.DateTime;

import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.engine.api.TaskService;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.cli.help.DateHelp.toLocalDate;

/**
 * Exposes the {@link TaskService}.
 *
 * @since Deployit 3.8
 */
@CliObject(name = "tasks")
@ClassHelp(description = "Access to the task engine of XL Deploy. !Deprecated! Use task2 instead.")
public class TaskClient extends DocumentedObject {

    private ProxiesInstance proxies;

    public TaskClient() {}

    public TaskClient(ProxiesInstance proxies) {
        this.proxies = proxies;
    }

    //
    // Method delegates to TaskService
    //

    @MethodHelp(description =
        "Lists the active tasks of the logged in user.",
        returns = "A list of tasks without step information")
    public List<TaskState> getMyCurrentTasks() {
        return proxies.getTaskRegistry().getMyCurrentTasks();
    }

    @MethodHelp(description =
        "Lists the active tasks of all users. Requires admin permission.",
        returns = "A list of tasks without step information")
    public List<TaskState> getAllCurrentTasks() {
        return proxies.getTaskRegistry().getAllCurrentTasks();
    }

    @MethodHelp(description =
        "Gets the task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") },
        returns = "The task without step information.")
    public TaskState get(String taskId) {
        return proxies.getTaskRegistry().getTask(taskId);
    }

    @MethodHelp(description =
        "Gets the task with steps.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") },
        returns = "The task, containing step information.")
    public TaskWithSteps steps(String taskId) {
        return proxies.getTaskRegistry().getSteps(taskId);
    }

    @MethodHelp(description =
        "Gets information about a step.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepId", description = "The position of the step in the step list.") },
        returns = "The step info.")
    public StepState step(String taskId, int stepId) {
        return proxies.getTaskRegistry().getStep(taskId, stepId, null);
    }

    @MethodHelp(description =
        "Starts a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void start(String taskId) {
        proxies.getTaskRegistry().start(taskId);
    }

    @MethodHelp(description = "Schedule a task for execution in the (near) future.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "dateTime", description = "The date and time of scheduled execution.") })
    public void schedule(String taskId, DateTime dateTime) {
        proxies.getTaskRegistry().schedule(taskId, dateTime);
    }

    @MethodHelp(description =
        "Gracefully stops an active task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void stop(String taskId) {
        proxies.getTaskRegistry().stop(taskId);
    }

    @MethodHelp(description =
        "Aborts an active task", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void abort(String taskId) {
        proxies.getTaskRegistry().abort(taskId);
    }

    @MethodHelp(description =
        "Cancels a stopped task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void cancel(String taskId) {
        proxies.getTaskRegistry().cancel(taskId);
    }

    @MethodHelp(description =
        "Archives an executed task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.") })
    public void archive(String taskId) {
        proxies.getTaskRegistry().archive(taskId);
    }
    
    @MethodHelp(description =
        "Skips steps of a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip.") },
        returns = "The updated task with step information.")
    public TaskWithSteps skip(String taskId, List<Integer> stepIds) {
        return proxies.getTaskRegistry().skip(taskId, newArrayList(stepIds));
    }

    @MethodHelp(description =
        "Unskips steps of a task.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip.") },
        returns = "The updated task with step information.")
    public TaskWithSteps unskip(String taskId, List<Integer> stepIds) {
        return proxies.getTaskRegistry().unskip(taskId, newArrayList(stepIds));
    }

    @MethodHelp(description = "Add a pause step to the task", parameters = {
            @ParameterHelp(name="taskId", description = "The id of the task."),
            @ParameterHelp(name="stepId", description = "The position at which the new pause step should appear.")
    })
    public TaskWithSteps addPause(String taskId, int stepId) {
        return proxies.getTaskRegistry().addPause(taskId, stepId);
    }

    @MethodHelp(description =
        "Assigns a pending task to another principal.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "owner", description = "The new task owner.") },
        returns = "The task without step information.")
    public TaskState assign(String taskId, String owner) {
        return proxies.getTaskRegistry().assign(taskId, owner);
    }

    @MethodHelp(description =
            "Takeover the ownership of a task from the current owner.", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "currentOwner", description = "The current task owner.") },
            returns = "The task without step information.")
    public TaskState takeover(String taskId, String currentOwner) {
        return proxies.getTaskRegistry().takeover(taskId, currentOwner);
    }

    @MethodHelp(description =
        "Gets all task information from the repository's archive in the specified date range.", parameters = {
            @ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format."),
            @ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format.") },
        returns = "A list of tasks without step information.")
    public List<TaskState> query(String begin, String end) {
        return proxies.getTaskRegistry().query(toLocalDate(begin), toLocalDate(end).plusDays(1)).collect(Collectors.toList());
    }

    @MethodHelp(description = "Gets all task information, including steps, from the repository's archive in the specified date range.", parameters = {
        @ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format."),
        @ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format.") },
        returns = "A list of tasks with their enclosed steps.")
    public List<TaskWithSteps> export(String begin, String end) {
        return proxies.getTaskRegistry().export(toLocalDate(begin), toLocalDate(end).plusDays(1)).collect(Collectors.toList());
    }
}
