package com.xebialabs.deployit.client;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.client.logger.Slf4jDeploymentListener;
import com.xebialabs.deployit.client.utils.Files;

import static java.lang.String.format;

public class DarPackager implements Packager {

	private final DeploymentListener listener;

	private final Manifest manifest = new Manifest();

	private final File workingDirectory;

	private final File darFile;

	public DarPackager(String application, String version, File targetDirectory, DeploymentListener listener) {
		this(application, version, targetDirectory, listener, version);
	}

	public DarPackager(String application, String version, File targetDirectory, DeploymentListener listener, String ciVersion) {
		this.listener = (listener != null ? listener : new Slf4jDeploymentListener(DeployitCli.class));
		this.workingDirectory = new File(targetDirectory, "deployit-working-dir");
		this.darFile = new File(targetDirectory, format("%s-%s.dar", application, version));
		final Attributes mainAttributes = manifest.getMainAttributes();
		mainAttributes.putValue("Manifest-Version", "1.0");
		mainAttributes.putValue("Deployit-Package-Format-Version", "1.3");
		mainAttributes.putValue("CI-Application", application);
		mainAttributes.putValue("CI-Version", ciVersion);
	}

	@Override
	public void addDeployable(Deployable deployable) {
		listener.debug("Add deployable " + deployable);
		final Map<String, Attributes> entries = manifest.getEntries();
		final Attributes attributes = new Attributes();

		attributes.putValue("CI-Type", deployable.getType());
		attributes.putValue("CI-Name", deployable.getName());
		for (Map.Entry<String, String> me : deployable.getValues().entrySet()) {
			attributes.putValue(me.getKey(), me.getValue());
		}
		entries.put(deployable.getEntryName(), attributes);

		if (!deployable.getTags().isEmpty()) {
			int idx = 0;
			for (String tag : deployable.getTags()) {
				attributes.putValue("CI-tags-EntryValue-" + idx, tag);
				idx++;
			}
		}

		if (deployable.getLocation() != null) {
			final File location = deployable.getLocation();
			if (location.isFile()) {
				listener.debug(format(" copy file %s to %s", location, workingDirectory));
				Files.copyFileTo(location, workingDirectory);

			} else {
				listener.debug(format(" copy dir  %s to %s", location, workingDirectory));
				Files.copyDirectoryTo(location, workingDirectory);
			}

			if (!location.getName().equals(deployable.getEntryName())) {
				File oldName = new File(workingDirectory, location.getName());
				File newName = new File(workingDirectory, deployable.getEntryName());
				listener.debug(format(" rename %s to %s", oldName, newName));
				Files.rename(oldName, newName);
			}
		}
	}

	@Override
	public void addDeployables(Deployable... deployables) {
		for (Deployable deployable : deployables) {
			addDeployable(deployable);
		}
	}

	@Override
	public void addDeployables(Collection<? extends Deployable> deployables) {
		for (Deployable deployable : deployables) {
			addDeployable(deployable);
		}
	}

    @Override
    public File perform() {
        if (darFile.exists()) {
            listener.info(String.format("dar file exists (%s), previously build by another phase, dot not recreate it ", darFile));
        } else {
            listener.info(String.format("dar file doest not exist (%s), go ", darFile));
            Files.jar(darFile, workingDirectory, generateManifestFile());
        }
        return darFile;
    }

    private File generateManifestFile() {
		final File meta_inf = new File(workingDirectory, "META-INF");
		meta_inf.mkdirs();
		File manifestFile = new File(meta_inf, "MANIFEST.MF");
		listener.info("Generate manifest file " + manifestFile.getAbsolutePath());
		FileOutputStream fos = null;
		try {
			dumpManifest();
			fos = new FileOutputStream(manifestFile);
			manifest.write(fos);
		} catch (IOException e) {
			throw new RuntimeException("generation of the manifest file failed", e);
		} finally {
			Closeables.closeQuietly(fos);
		}
		return manifestFile;
	}

	private void dumpManifest() throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			manifest.write(baos);
		} finally {
			Closeables.closeQuietly(baos);
		}
		listener.debug(new String(baos.toByteArray()));
	}


}
