package com.xebialabs.deployit.documentation;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.api.boot.PluginBooter;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.*;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Generates markdown documentation for CIs found on the classpath.
 */
public class CiReferenceHtmlGenerator {

    static {
        PluginBooter.bootWithoutGlobalContext();
    }

    private List<String> prefixes;
    private PrintWriter writer;

    public CiReferenceHtmlGenerator(List<String> prefixes, PrintWriter writer) {
        this.prefixes = prefixes;
        this.writer = writer;
    }

    public void generate() {
        logger.info("Generating documentation for " + (!prefixes.isEmpty() ? "CIs with prefix '" + prefixes + "'" : "all CIs"));
        List<Descriptor> cis = findDescriptorsToGenerateReferenceDocFor(prefixes);
        logger.info("Found " + cis.size() + " CI descriptors to generate documentation for.");
        writer.println("<div style='width:100%; position:relative'>");
        generateCiTableOfContentForAllCategories(cis);
        generateCiDetails(cis);
        writer.println("</div>");
    }


    private void generateCiTableOfContentForAllCategories(Collection<Descriptor> cis) {
        writer.printf("<h1 class='ci-toc-category'>CI Reference</h1>");
        writer.printf("<h2>Configuration Item Overview</h2>");
        generateCiTableOfContentForCategory(cis, "Deployable Configuration Items", DEPLOYABLE_PREDICATE);
        generateCiTableOfContentForCategory(cis, "Deployed Configuration Items", DEPLOYED_PREDICATE);
        generateCiTableOfContentForCategory(cis, "Topology Configuration Items", TOPOLOGY_PREDICATE);

        generateCiTableOfContentForCategory(cis, "Virtual Deployable Configuration Items", VIRTUAL_DEPLOYABLE_PREDICATE);
        generateCiTableOfContentForCategory(cis, "Virtual Deployed Configuration Items", VIRTUAL_DEPLOYED_PREDICATE);
        generateCiTableOfContentForCategory(cis, "Virtual Topology Configuration Items", VIRTUAL_TOPOLOGY_PREDICATE);

    }

    private void generateCiDetails(Collection<Descriptor> cis) {
        writer.printf("<h2 class='ci-details-title'>Configuration Item Details</h2>");

        for (Descriptor ci : cis) {
            writer.printf("<a name='%s'></a><h3 class='ci-detail-title'>%s</h3>", ci.getType(), ci.getType());
            if (!ci.getSuperClasses().isEmpty()) {
                writer.print("<div><table class='ci-relations-table'>");
                if (!ci.getSuperClasses().isEmpty())
                    generateTypeRow(ci.getSuperClasses(), "Hierarchy", "<span> >> </span>");

                if (!ci.getInterfaces().isEmpty())
                    generateTypeRow(ci.getInterfaces(), "Interfaces", "<span>, </span>");
                writer.print("</table></div>");
            }
            writer.printf("<p class='ci-description'>%s</p>\n", ci.getDescription());

            Collection<PropertyDescriptor> propertyDescriptors = sortProperties(ci.getPropertyDescriptors());
            generateCiPropertiesTable(propertyDescriptors, "Public Properties", PUBLIC_PROPERTY_PREDICATE);
            generateCiPropertiesTable(propertyDescriptors, "Hidden Properties", HIDDEN_PROPERTY_PREDICATE);
            writer.println("<hr class='ci-details-separator'/>");
        }
    }

    private void generateTypeRow(Collection<Type> types, String tHeader, String separator) {
        writer.printf("<tr><th>%s</th><td>", tHeader);
        int i = 0;
        for (Type iType : types) {
            writeTypeAnchor(iType);
            i++;
            if (i < types.size()) {
                writer.print(separator);
            }
        }
        writer.print("</td></tr>");
    }

    private void writeTypeAnchor(Type type) {
        if (isTypePartOfReferencedDoc(type)) {
            writer.printf("<a href='#%s'>%s</a>", type, type);
        } else {
            writer.printf("<span>%s</span>", type);
        }
    }

    private void generateCiPropertiesTable(Collection<PropertyDescriptor> pds, String category, Predicate<PropertyDescriptor> predicate) {
        Collection<PropertyDescriptor> filtered = Collections2.filter(pds, predicate);
        if (filtered.isEmpty()) {
            return;
        }

        writer.printf("<table class='ci-properties nobreak'  cellspacing='0' cellpadding='3'><tr class='odd ci-prop-header'><th>%s</th></tr>", category);
        int i = 0;
        for (PropertyDescriptor pd : filtered) {
            String oddEven = i % 2 == 0 ? "even" : "odd";
            String required = pd.isRequired() ? "*" : "";
            String kind = pd.getKind().toString();
            if (pd.getKind() == PropertyKind.ENUM) {
                kind = kind + " " + Arrays.toString(pd.getEnumValues().toArray());
            }

            if (pd.getKind() == PropertyKind.CI || pd.getKind() == PropertyKind.SET_OF_CI) {
                kind = kind + "&lt;" + pd.getReferencedType() + "&gt;";
            }

            String defaultValue = "";
            if (pd.getDefaultValue() != null) {
                defaultValue = "&nbsp;=&nbsp;<span class='ci-property-default'>" + pd.getDefaultValue() + "</span>";
            }

            writer.printf("<tr class='%s'><td><a name='%s_%s'></a><span class='ci-property-name'>%s</span><span class='ci-property-req'>%s</span>&nbsp;:&nbsp;<span class='ci-property-kind'>%s</span>%s<p class='ci-property-desc'>%s</p></td>\n",
                    oddEven, pd.getDeclaringDescriptor().getType(),pd.getName(), pd.getName(), required, kind, defaultValue, pd.getDescription());
            i++;
        }
        writer.println("</table>");
    }


    private void generateCiTableOfContentForCategory(Collection<Descriptor> cis, String category, Predicate<Descriptor> predicate) {
        Collection<Descriptor> filtered = Collections2.filter(cis, predicate);
        if (!filtered.isEmpty()) {
            writer.printf("<h3 class='ci-toc-category'>%s</h3>", category);
            generateCiTableOfContent(filtered, writer);
        }
    }

    private void generateCiTableOfContent(Collection<Descriptor> cis, PrintWriter writer) {
        writer.println("<table class='ci-toc nobreak'  cellspacing='0' cellpadding='3'><tr class='odd ci-toc-header'><th>CI</th><th>Description</th></tr>");
        int i = 0;
        for (Descriptor ci : cis) {
            String oddEven = i % 2 == 0 ? "even" : "odd";
            writer.printf("<tr class='%s'><td><a href='#%s'>%s</a></td><td>%s</td>\n", oddEven, ci.getType(), ci.getType(), extractFirstSentence(ci.getDescription()));
            i++;
        }
        writer.println("</table>");
    }

    private String extractFirstSentence(String s) {
        if (s == null) {
            return "";
        }
        if (s.indexOf(".") > -1) {
            return s.substring(0, s.indexOf("."));
        }
        return s;
    }

    private List<Descriptor> findDescriptorsToGenerateReferenceDocFor(final List<String> prefixes) {
        Collection<Descriptor> filtered = Collections2.filter(DescriptorRegistry.getDescriptors(), new Predicate<Descriptor>() {
            @Override
            public boolean apply(Descriptor input) {
                return isTypePartOfReferencedDoc(input.getType());
            }
        });

        List<Descriptor> needsRefDoc = newArrayList(filtered);

        Collections.sort(needsRefDoc, new Comparator<Descriptor>() {
            @Override
            public int compare(Descriptor o1, Descriptor o2) {
                return o1.getType().toString().compareTo(o2.getType().toString());
            }
        });

        return needsRefDoc;
    }

    private boolean isTypePartOfReferencedDoc(Type type) {
        return prefixes.isEmpty() || prefixes.contains(type.getPrefix());
    }

    private void appendToFile(String outputFilename, byte[] data) throws IOException {
        FileOutputStream fos = new FileOutputStream(outputFilename, true);
        try {
            fos.write(data);
        } finally {
            Closeables.closeQuietly(fos);
        }
    }

    private Collection<PropertyDescriptor> sortProperties(Collection<PropertyDescriptor> propertyDescriptors) {
        List<PropertyDescriptor> result = newArrayList(propertyDescriptors);
        Collections.sort(result, new Comparator<PropertyDescriptor>() {
            @Override
            public int compare(PropertyDescriptor o1, PropertyDescriptor o2) {
                int result = o1.isRequired() == o2.isRequired() ? 0 : (o1.isRequired() ? -1 : 1);
                if (result == 0) {
                    result = o1.getName().compareTo(o2.getName());
                }
                return result;
            }
        });
        return result;
    }

    private static final Predicate<Descriptor> DEPLOYABLE_PREDICATE = new Predicate<Descriptor>() {
        @Override
        public boolean apply(Descriptor input) {
            return !input.isVirtual() && input.isAssignableTo(Deployable.class);
        }
    };

    private static final Predicate<Descriptor> VIRTUAL_DEPLOYABLE_PREDICATE = new Predicate<Descriptor>() {
        @Override
        public boolean apply(Descriptor input) {
            return input.isVirtual() && input.isAssignableTo(Deployable.class);
        }
    };

    private static final Predicate<Descriptor> DEPLOYED_PREDICATE = new Predicate<Descriptor>() {
        @Override
        public boolean apply(Descriptor input) {
            return !input.isVirtual() && input.isAssignableTo(Deployed.class);
        }
    };

    private static final Predicate<Descriptor> VIRTUAL_DEPLOYED_PREDICATE = new Predicate<Descriptor>() {
        @Override
        public boolean apply(Descriptor input) {
            return input.isVirtual() && input.isAssignableTo(Deployed.class);
        }
    };

    private static final Predicate<Descriptor> TOPOLOGY_PREDICATE = new Predicate<Descriptor>() {
        @Override
        public boolean apply(Descriptor input) {
            return !input.isVirtual() && !input.isAssignableTo(Deployable.class) && !input.isAssignableTo(Deployed.class);
        }
    };

    private static final Predicate<Descriptor> VIRTUAL_TOPOLOGY_PREDICATE = new Predicate<Descriptor>() {
        @Override
        public boolean apply(Descriptor input) {
            return input.isVirtual() && !input.isAssignableTo(Deployable.class) && !input.isAssignableTo(Deployed.class);
        }
    };

    private static final Predicate<PropertyDescriptor> HIDDEN_PROPERTY_PREDICATE = new Predicate<PropertyDescriptor>() {
        @Override
        public boolean apply(PropertyDescriptor input) {
            return input.isHidden();
        }
    };

    private static final Predicate<PropertyDescriptor> PUBLIC_PROPERTY_PREDICATE = new Predicate<PropertyDescriptor>() {
        @Override
        public boolean apply(PropertyDescriptor input) {
            return !input.isHidden();
        }
    };


    private static final Logger logger = LoggerFactory.getLogger(CiReferenceHtmlGenerator.class);
}
