package com.xebialabs.deployit.documentation;

import com.google.common.collect.ForwardingMap;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.google.common.io.InputSupplier;
import com.samskivert.mustache.Mustache;
import com.samskivert.mustache.MustacheException;
import com.xebialabs.overthere.RuntimeIOException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.URL;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import static com.google.common.collect.Maps.newHashMap;

public class IOUtils {

    private static InputSupplier<InputStream> newUrlInputSupplier(final URL url) {
        return new InputSupplier<InputStream>() {
            @Override
            public InputStream getInput() throws IOException {
                return url.openStream();
            }
        };
    }

    public static void copy(URL fromUrl, File toFile) {
        try {
            Files.copy(IOUtils.newUrlInputSupplier(fromUrl), toFile);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    public static String getText(URL url) {
        InputStream is = null;
        try {
            is = url.openStream();
            return new String(ByteStreams.toByteArray(is));
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            Closeables.closeQuietly(is);
        }
    }

    public static String getText(File file) {
        try {
            return new String(Files.toByteArray(file));
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }


    public static void copyDirectory(File sourceLocation, File targetLocation) {

        if (sourceLocation.isDirectory()) {
            if (!targetLocation.exists()) {
                targetLocation.mkdir();
            }

            for (String child : sourceLocation.list()) {
                copyDirectory(new File(sourceLocation, child), new File(targetLocation, child));
            }
        } else {
            try {
                Files.copy(sourceLocation, targetLocation);
            } catch (IOException e) {
                throw new RuntimeIOException(e);
            }
        }
    }

    public static String replacePlaceholders(String replaceable, ContextProperties values) {
        StringWriter writer = new StringWriter();
        replacePlaceholders(new StringReader(replaceable), values, writer);
        return writer.toString();
    }

    public static void replacePlaceholders(Reader replaceable, ContextProperties values, Writer writer) {
        try {
            if (values == null || values.isEmpty()) {
                copy(replaceable, writer);
            }

            Mustache.compiler().compile(replaceable).execute(new MustacheContext((Map)values), writer);
        } catch (MustacheException me) {
            throw new RuntimeException("Could not replace keys in " + replaceable, me);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private static int copy(Reader input, Writer output) throws IOException {
        char[] buffer = new char[1024 * 4];
        int count = 0;
        int n = 0;
        while (-1 != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
            count += n;
        }
        return count;
    }

    public static File createUniqueFileNameWithNewExtension(File source, File workingDir, String ext) {
        String name = extractFileNameWithoutExtension(source);
        String newFileName = name + "." + ext;
        for (int i = 1; i < 50; i++) {
            File newFile = new File(workingDir, newFileName);
            if (!newFile.exists()) {
                return newFile;
            }
            newFileName = name + i + "." + ext;
        }
        throw new RuntimeException("Cannot generate unique file name for '" + name + "." + ext + "' in directory " + workingDir.getAbsolutePath());
    }

    public static File createFileNameWithNewExtension(File source, File workingDir, String ext) {
        String name = extractFileNameWithoutExtension(source);
        String newFileName = name + "." + ext;
        File newFile = new File(workingDir, newFileName);
        if (newFile.exists()) {
            newFile.delete();
            try {
                newFile.createNewFile();
            } catch (IOException e) {
                throw new RuntimeIOException();
            }
        }
        return newFile;
    }

    public static String extractFileNameWithoutExtension(File file) {
        String fileName = file.getName();
        int index = fileName.lastIndexOf('.');
        if (index > -1) {
            fileName = fileName.substring(0, index);
        }
        return fileName;
    }

    public static String extractFileName(URL url) {
        String urlAsString = url.toString();
        if (urlAsString.endsWith("/")) {
            urlAsString = urlAsString.substring(0, urlAsString.length() - 1);
        }

        int slashIndex = urlAsString.lastIndexOf('/');
        return urlAsString.substring(slashIndex + 1);
    }

    public static String extractFileNameExtension(String fileName) {
        int index = fileName.lastIndexOf('.');
        if (index > -1) {
            return fileName.substring(index + 1);
        }
        return "";
    }

    public static File explodeArchive(File archive, File explodedFolder) {
        try {
            final ZipInputStream zipEntryStream = new ZipInputStream(new FileInputStream(archive));
            if (!explodedFolder.exists()) {
                explodedFolder.mkdir();
            }
            for (; ; ) {
                ZipEntry entry = zipEntryStream.getNextEntry();
                if (entry == null) {
                    return explodedFolder;
                }

                try {

                    if (entry.isDirectory()) {
                        final File file = new File(explodedFolder, entry.getName());
                        if (!file.exists()) {
                            isTrue(file.mkdirs(), "Could not create directory: " + entry.getName());
                        }
                        continue;
                    }

                    final File destinationFile = new File(explodedFolder, entry.getName());
                    final String parentDirPath = destinationFile.getParent();
                    if (parentDirPath != null) {
                        final File destinationDir = new File(parentDirPath);
                        if (!destinationDir.exists()) {
                            isTrue(destinationDir.mkdirs(), "Could not create directory: " + entry.getName());
                        }
                    }
                    if (!destinationFile.exists())
                        isTrue(destinationFile.createNewFile(), "Could not create file: " + entry.getName());
                    ByteStreams.copy(zipEntryStream, new FileOutputStream(destinationFile));
                } finally {
                    zipEntryStream.closeEntry();
                }
            }
        } catch (IOException exc) {
            // On exception, clean up!
            try {
                Files.deleteRecursively(explodedFolder);
            } catch (Exception e) {
                logger.error("Could not delete {}", explodedFolder, e);
            }
            throw new RuntimeIOException(exc);
        }
    }

    private static void isTrue(boolean expression, String message) {
        if (!expression) {
            throw new IllegalArgumentException(message);
        }
    }

    private static class MustacheContext extends ForwardingMap<String, Object> {

        private Map<String, Object> delegate = newHashMap();

        private MustacheContext(Map<String, Object> delegate) {
            this.delegate = delegate;
        }

        @Override
        protected Map<String, Object> delegate() {
            return delegate;
        }

        @Override
        public Object get(Object key) {
            if (delegate.containsKey(key)) {
                return delegate.get(key);
            }

            String prefix = key + ".";
            Map<String, Object> map = newHashMap();
            for (String fullKey : delegate.keySet()) {
                if (fullKey.startsWith(prefix)) {
                    map.put(fullKey.substring(prefix.length()), delegate.get(fullKey));
                }
                if (!map.isEmpty()) {
                    return new MustacheContext(map);
                }
            }

            return null;
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(IOUtils.class);
}