/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api.execution;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.joda.time.DateTime;

import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 * Information about a step.
 *
 * A step is a concrete action to be performed to accomplish a task. XL Deploy ships with many step implementations
 * for common actions. Other, middleware-specific steps are contributed by the plugins.
 */
@PublicApiRef
public interface StepState {
    /**
     * @return Textual description of the step.
     */
    String getDescription();

    /**
     * @return The current state of the step.
     */
    StepExecutionState getState();

    /**
     * @return The entire log output of the last attempt of the step. This may be long.
     */
    String getLog();

    /**
     * @return Lists the entire logs of all previous attempts of this step, if recorded; or null
     */
    List<String> getPreviousAttemptsLogs();

    /**
     * @return Lists the entire logs of all attempts. Combines getLog() and previousAttemptsLogs(). Never null.
     */
    default List<String> getLogs() {
        List<String> prevLogs = getPreviousAttemptsLogs();
        List<String> result = prevLogs == null ? new ArrayList<>() : new ArrayList<>(getPreviousAttemptsLogs());
        if (getLog() != null) result.add(getLog());
        return result;
    }

    /**
     * @return The date when the step was started.
     */
    DateTime getStartDate();

    /**
     * @return The date when the step was completed.
     */
    DateTime getCompletionDate();

    /**
     * @return The number of times this step has failed.
     */
    int getFailureCount();

    /**
     * @return A map with step-specific entries (for example, the path to a script).
     */
    Map<String, String> getMetadata();

    boolean getSkippable();
}
