/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api.dto;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import jakarta.ws.rs.FormParam;
import jakarta.ws.rs.core.MediaType;
import org.jboss.resteasy.annotations.providers.multipart.PartType;

import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

/**
 * The parameter object for certain REST API and it represents both the artifact CI and its data.
 * <p/>
 * Artifacts are uploaded to the XL Deploy Server as a multipart HTTP request.
 * It consists of the following parts:
 * <dl>
 * <dt>configurationItem</dt>
 * <dd>The property data of the artifact in XML. See {@link com.xebialabs.deployit.plugin.api.udm.artifact.Artifact} for details.</dd>
 * <dd>Content-type: application/xml</dd>
 * <p/>
 * <dt>filename</dt>
 * <dd>The file name of the uploaded artifact.</dd>
 * <dd>Content-type: application/xml</dd>
 * <p/>
 * <dt>fileData</dt>
 * <dd>The artifact data.</dd>
 * <dd>Content-type: application/octet-stream</dd>
 * </dl>
 */
@PublicApiRef
public class ArtifactAndData extends AbstractDto {

    @FormParam("configurationItem")
    @PartType(MediaType.APPLICATION_XML)
    private Artifact artifact;

    @FormParam("filename")
    @PartType(MediaType.APPLICATION_XML)
    private String filename;

    @FormParam("fileData")
    @PartType(MediaType.APPLICATION_OCTET_STREAM)
    private File file;

    @FormParam("fileStream")
    @PartType(MediaType.APPLICATION_OCTET_STREAM)
    private InputStream data;

    public ArtifactAndData() {
    }

    /**
     * @param artifact The CI representing the artifact.
     * @param file     The artifact itself as an file.
     */
    public ArtifactAndData(Artifact artifact, File file) {
        this.artifact = artifact;
        this.file = file;
    }

    /**
     * @param artifact The CI representing the artifact.
     * @param filename The file name of the uploaded artifact.
     * @param data     The artifact itself as an input stream.
     * @deprecated Use {@link ArtifactAndData#ArtifactAndData(com.xebialabs.deployit.plugin.api.udm.artifact.Artifact, java.io.File)}
     */
    @Deprecated
    public ArtifactAndData(Artifact artifact, String filename, InputStream data) {
        this.artifact = artifact;
        this.filename = filename;
        this.data = data;
    }

    /**
     * @param artifact The CI representing the artifact.
     * @param filename The file name of the uploaded artifact.
     * @param data     The artifact itself as an array of bytes.
     * @deprecated Use {@link ArtifactAndData#ArtifactAndData(com.xebialabs.deployit.plugin.api.udm.artifact.Artifact, java.io.File)}
     */
    @Deprecated
    public ArtifactAndData(Artifact artifact, String filename, byte[] data) {
        this(artifact, filename, new ByteArrayInputStream(data));
    }

    /**
     * @param artifact The CI representing the artifact.
     * @param data     The artifact itself as an input stream.
     * @deprecated Use {@link ArtifactAndData#ArtifactAndData(com.xebialabs.deployit.plugin.api.udm.artifact.Artifact, java.io.File)}
     */
    @Deprecated
    public ArtifactAndData(Artifact artifact, InputStream data) {
        this(artifact, null, data);
    }

    /**
     * @param artifact The CI representing the artifact.
     * @param data     The artifact itself as an array of bytes.
     * @deprecated Use {@link ArtifactAndData#ArtifactAndData(com.xebialabs.deployit.plugin.api.udm.artifact.Artifact, java.io.File)}
     */
    @Deprecated
    public ArtifactAndData(Artifact artifact, byte[] data) {
        this(artifact, null, data);
    }

    /**
     * @return The CI representing the artifact.
     */
    public Artifact getArtifact() {
        return artifact;
    }

    /**
     * @param artifact The CI representing the artifact.
     */
    public void setArtifact(Artifact artifact) {
        this.artifact = artifact;
    }

    /**
     * @return The artifact data as an input stream.
     */
    public InputStream getDataInputStream() {
        if (data == null && file != null) {
            try {
                data = new BufferedInputStream(new FileInputStream(file));
            }
            catch (FileNotFoundException e) {
                throw new IllegalStateException("Could not read the artifact file");
            }
        }

        return data;
    }

    /**
     * @param data The artifact data as an input stream.
     */
    public void setDataInputStream(final InputStream data) {
        this.data = data;
    }

    /**
     * @return The artifact data as an array of bytes.
     * @deprecated Use com.xebialabs.deployit.engine.api.dto.ArtifactAndData#getDataInputStream()
     */
    @Deprecated
    public byte[] getData() {
        final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        write(getDataInputStream(), byteArrayOutputStream);
        return byteArrayOutputStream.toByteArray();
    }

    /**
     * @param data The artifact data as an array of bytes.
     * @deprecated Use com.xebialabs.deployit.engine.api.dto.ArtifactAndData#setDataInputStream(java.io.InputStream)
     */
    @Deprecated
    public void setData(final byte[] data) {
        this.data = new ByteArrayInputStream(data);
    }

    /**
     * @return The file name of the uploaded artifact.
     */
    public String getFilename() {
        return filename;
    }

    /**
     * @param filename The file name of the uploaded artifact.
     */
    public void setFilename(String filename) {
        this.filename = filename;
    }


    private static void write(InputStream from, OutputStream to) {
        try {
            byte[] bytes = new byte[1024 * 128];
            int nRead;
            while ((nRead = from.read(bytes, 0, bytes.length)) != -1) {
                to.write(bytes, 0, nRead);
            }
        } catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }
    }
}
