/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api.dto;

import com.xebialabs.deployit.plugin.api.deployment.ResolvedPlaceholder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.documentation.PublicApiRef;

import java.util.*;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

/**
 * The parameter object of a deployment action.
 */
@PublicApiRef
public class Deployment extends AbstractDto {

    /**
     * The type of deployment: INITIAL, UPDATE or UNDEPLOYMENT.
     */
    @PublicApiRef
    public enum DeploymentType {
        INITIAL, UPDATE, UNDEPLOYMENT
    }

    private String id;

    private ConfigurationItem deployedApplication;

    private List<ConfigurationItem> deployeds = new ArrayList<>();

    private List<ConfigurationItemId> deployables = new ArrayList<>();

    private List<ConfigurationItemId> containers = new ArrayList<>();

    private List<Deployment> requiredDeployments = new ArrayList<>();

    private Set<ResolvedPlaceholder> resolvedPlaceholders = new HashSet<>();

    private int deploymentGroupIndex;

    private DeploymentType deploymentType;

    /**
     * @return The id of the deployment action.
     */
    public String getId() {
        return id;
    }

    /**
     * @param id The id of the deployment action.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return The application being deployed, upgraded or undeployed.
     */
    public ConfigurationItem getDeployedApplication() {
        return deployedApplication;
    }

    /**
     * @return A list of {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}s representing the deployeds of this deployment.
     */
    public List<ConfigurationItem> getDeployeds() {
        return deployeds;
    }

    /**
     * @return The {@link DeploymentType} of the current deployment.
     */
    public DeploymentType getDeploymentType() {
        return deploymentType;
    }

    /**
     * @param deploymentType The {@link DeploymentType} of the current deployment.
     */
    public void setDeploymentType(DeploymentType deploymentType) {
        this.deploymentType = deploymentType;
    }

    /**
     * @param type The {@link DeploymentType} to check against.
     * @return Whether the current deployment has the given {@link DeploymentType}.
     */
    public boolean isOfType(DeploymentType type) {
        return this.deploymentType == type;
    }

    /**
     * @param deployedApplication The application being deployed, upgraded or undeployed.
     */
    public void setDeployedApplication(ConfigurationItem deployedApplication) {
        this.deployedApplication = deployedApplication;
    }

    /**
     * @param deployeds A list of {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}s representing the deployeds of this deployment.
     */
    public void setDeployeds(List<ConfigurationItem> deployeds) {
        this.deployeds = deployeds;
    }

    /**
     * Adds a {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} to the list of deployeds of this deployment.
     */
    public void add(ConfigurationItem object) {
        this.deployeds.add(object);
    }

    /**
     * Adds a list of {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem}s to the list of deployeds of this deployment.
     */
    public void addAll(List<ConfigurationItem> deployeds) {
        this.deployeds.addAll(deployeds);
    }

    /**
     * @return The CI id of the deployed package.
     */
    public String getVersion() {
        return (String) deployedApplication.getProperty("version");
    }

    /**
     * @return The CI id of the deployment environment.
     */
    public String getEnvironment() {
        return (String) deployedApplication.getProperty("environment");
    }

    /**
     * @return A list of CI references ({@link ConfigurationItemId}s) representing the deployables of this deployment.
     */
    public List<ConfigurationItemId> getDeployables() {
        return deployables;
    }

    /**
     * @param deployables A list of CI references ({@link ConfigurationItemId}s) representing the deployables of this deployment.
     */
    public void setDeployables(List<ConfigurationItemId> deployables) {
        this.deployables = deployables;
    }

    /**
     * @return A list of CI references ({@link ConfigurationItemId}s) representing the containers of this deployment.
     */
    public List<ConfigurationItemId> getContainers() {
        return containers;
    }

    /**
     * @param containers A list of CI references ({@link ConfigurationItemId}s) representing the containers of this deployment.
     */
    public void setContainers(List<ConfigurationItemId> containers) {
        this.containers = containers;
    }

    /**
     * @return A list of Deployments ({@link com.xebialabs.deployit.engine.api.dto.Deployment}s) representing the dependent deployments.
     */
    public List<Deployment> getRequiredDeployments() {
        return this.requiredDeployments;
    }

    public List<List<Deployment>> getGroupedRequiredDeployments() {
        Map<Integer, List<Deployment>> collect = this.requiredDeployments.stream().collect(groupingBy(Deployment::getDeploymentGroupIndex));
        List<Integer> keys = new ArrayList<>(collect.keySet());
        Collections.sort(keys);
        return keys.stream().map(collect::get).collect(toList());
    }

    /**
     * @param dependencies A list of Deployments ({@link com.xebialabs.deployit.engine.api.dto.Deployment}s) representing the dependent deployments.
     */
    public void setGroupedRequiredDeployments(List<List<Deployment>> dependencies) {
        for (int i = 0; i < dependencies.size(); i++) {
            final int index = i;
            dependencies.get(i).forEach(x -> x.setDeploymentGroupIndex(index + 1));
        }
        this.requiredDeployments = dependencies.stream().flatMap(Collection::stream).collect(toList());
    }

    public int getDeploymentGroupIndex() {
        return deploymentGroupIndex;
    }

    public void setDeploymentGroupIndex(int deploymentGroupIndex) {
        this.deploymentGroupIndex = deploymentGroupIndex;
    }

    public void setRequiredDeployments(List<Deployment> requiredDeployments) {
        this.requiredDeployments = requiredDeployments;
    }

    public Set<ResolvedPlaceholder> getResolvedPlaceholders() {
        return resolvedPlaceholders;
    }

    public void addResolvedPlaceholders(Set<ResolvedPlaceholder> resolvedPlaceholders) {
        this.resolvedPlaceholders.addAll(resolvedPlaceholders);
    }

    // for python wrapper
    public void setResolvedPlaceholders(Set<ResolvedPlaceholder> resolvedPlaceholders) {
        this.resolvedPlaceholders = resolvedPlaceholders;
    }
}
