package com.xebialabs.deployit.engine.api;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

/**
 * Manages permissions in Deployit.
 */
@Path("/security")
@Consumes({ MediaType.APPLICATION_XML })
@Produces({ MediaType.APPLICATION_XML })
public interface PermissionService {

    /**
     * Checks if a permission is granted to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     * @param role the role to which the permission should be granted.
     * @return {@code true} if granted.
     */
    @GET
    @Path("permission/{permission}/{role}/{id:.*?}")
    boolean isGranted(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Grants a permission to a role on a CI.
     *
     * @permission security#edit
     * @param permission the name of the permission to grant.
     * @param id the path of the CI to grant the permission on.
     * @param role the role to which the permission should be granted.
     */
    @PUT
    @Path("permission/{permission}/{role}/{id:.*?}")
    void grant(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Revokes the permission of a role on a CI.
     *
     *
     * @permission security#edit
     * @param permission the name of the permission to revoke.
     * @param id the path of the CI to revoke the permission from.
     * @param role the role from which the permission should be revoked.
     */
    @DELETE
    @Path("permission/{permission}/{role}/{id:.*?}")
    void revoke(@PathParam("permission") String permission, @PathParam("id") String id, @PathParam("role") String role);

    /**
     * Checks if the currently logged in user has a certain permission on a CI. If the user does not have this permission, error code {@code 403} is returned.
     *
     * @param permission the name of the permission to check.
     * @param id the path of the CI to check the permission on.
     */
    @HEAD
    @Path("check/{permission}/{id:.*?}")
    void checkMyPermission(@PathParam("permission") String permission, @PathParam("id") String id);
}
