package com.xebialabs.deployit.engine.api;

import java.util.List;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;

/**
 * Manages tasks on the Deployit Server.
 */
@Path("/task")
@Consumes(MediaType.APPLICATION_XML)
@Produces(MediaType.APPLICATION_XML)
public interface TaskService {

    /**
     * Returns the active tasks of the logged in user.
     *
     * @return a list of tasks.
     */
    @GET
    @Path("current")
    List<TaskState> getMyCurrentTasks();

    /**
     * Returns all active tasks for all users. Only allowed for admin.
     *
     * @return a list of tasks.
     */
    @GET
    @Path("current/all")
    List<TaskState> getAllCurrentTasks();

    /**
     * Returns a task by ID.
     *
     * @param taskId the ID of the task
     * @return the task.
     */
    @GET
    @Path("{taskid}")
    TaskState getTask(@PathParam("taskid") String taskId);

    /**
     * Returns a task containing step information.
     *
     * @param taskId the ID of the task
     * @return a task with full step information.
     */
    @GET
    @Path("{taskid}/step")
    TaskWithSteps getSteps(@PathParam("taskid") String taskId);

    /**
     * Retrieves information about a step.
     *
     * @param taskId          the ID of the task
     * @param stepId          the ordinal number of the step, starting from 1
     * @param ifModifiedSince the if-modified-since date in <a href="http://datatracker.ietf.org/doc/rfc1123/">RFC 1123</a> (<a href="http://datatracker.ietf.org/doc/rfc822/">RFC 822</a> with 4-digit years) date format.
     * @return if the step has not been modified since {@code ifModifiedSince}, a response with status code of 304 (Not Modified), otherwise a
     *         response with a status code of 200 (OK) and XML containing a StepState.
     */
    @GET
    @Path("{taskid}/step/{stepId}")
    StepState getStep(@PathParam("taskid") String taskId, @PathParam("stepId") int stepId, @HeaderParam("If-Modified-Since") DateTime ifModifiedSince);

    /**
     * Starts a task.
     *
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/start")
    void start(@PathParam("taskid") String taskId);

    /**
     * Gracefully stops an active task.
     *
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/stop")
    void stop(@PathParam("taskid") String taskId);

    /**
     * Aborts an active task.
     *
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/abort")
    void abort(@PathParam("taskid") String taskId);

    /**
     * Cancels a stopped task.
     *
     * @param taskId the ID of the task
     */
    @DELETE
    @Path("{taskid}")
    void cancel(@PathParam("taskid") String taskId);

    /**
     * Archive an executed task.
     *
     * @param taskId the ID of the task
     */
    @POST
    @Path("{taskid}/archive")
    void archive(@PathParam("taskid") String taskId);

    /**
     * Indicates that one or more steps should be skipped.
     *
     * @permission task#skip_step
     * @param taskId the ID of the task
     * @param stepIds the IDs of the steps to skip
     * @return a task with full step information.
     */
    @POST
    @Path("{taskid}/skip")
    TaskWithSteps skip(@PathParam("taskid") String taskId, List<Integer> stepIds);

    /**
     * Indicates that one or more steps should no longer be skipped, but executed.
     *
     * @permission task#skip_step
     * @param taskId the ID of the task
     * @param stepIds the IDs of the steps to unskip
     * @return a task with full step information.
     */
    @POST
    @Path("{taskid}/unskip")
    TaskWithSteps unskip(@PathParam("taskid") String taskId, List<Integer> stepIds);

    /**
     * Moves a step.
     *
     * @permission task#move_step
     * @param taskId the ID of the task
     * @param stepId the current position of the step in the step list.
     * @param newPosition the new position of the step in the step list.
     * @return a task with full step information.
     */
    @POST
    @Path("{taskid}/move/{stepId}/{position}")
    TaskWithSteps moveStep(@PathParam("taskid") String taskId, @PathParam("stepId") int stepId, @PathParam("position") int newPosition);

    /**
     * Add a pause step at the specified position.
     *
     * @param taskId the ID of the task
     * @param stepId the position of the step in the step list.
     * @return a task with full step information.
     */
    @POST
    @Path("{taskid}/pause/{stepId}")
    TaskWithSteps addPause(@PathParam("taskid") String taskId, @PathParam("stepId") int stepId);

    /**
     * Assigns a task to a different user.
     *
     * @permission task#assign
     * @param taskId the ID of the task
     * @param owner the name of the user that will be the new owner of the task.
     * @return the task.
     */
    @POST
    @Path("{taskid}/assign/{owner}")
    TaskState assign(@PathParam("taskid") String taskId, @PathParam("owner") String owner);

    /**
     * Searches for tasks without step information.
     *
     * @param begin The first day we want to see tasks from, or {@code null} for no such limit.
     * @param end The last day we want to see tasks from, or {@code null} for no such limit.
     * @return a list of tasks.
     */
    @GET
    @Path("query")
    List<TaskState> query(@QueryParam("begindate") LocalDate begin, @QueryParam("enddate") LocalDate end);

    /**
     * Searches for tasks with detailed step information.
     *
     * @param begin The first day we want to see tasks from, or {@code null} for no such limit.
     * @param end The last day we want to see tasks from, or {@code null} for no such limit.
     * @return a list of tasks with step information.
     */
    @GET
    @Path("export")
    List<TaskWithSteps> export(@QueryParam("begindate") LocalDate begin, @QueryParam("enddate") LocalDate end);
}
