package com.xebialabs.deployit.engine.api;

import java.util.List;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

import org.jboss.resteasy.annotations.providers.multipart.MultipartForm;

import com.xebialabs.deployit.engine.api.dto.FileUpload;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

/**
 * Manages Deployment Packages.
 */
@Path("/package")
@Consumes({MediaType.APPLICATION_XML})
@Produces({MediaType.APPLICATION_XML})
public interface PackageService {

    /**
     * Lists all {@link ImportablePackage application packages} that are present in the {@code importablePackages} directory on the Deployit Server.
     *
     * @return List of names of packages that can be imported.
     */
    @GET
    @Path("import")
    List<String> list();

    /**
     * Imports an {@link ImportablePackage application package} that is present in the {@code importablePackages} directory on the Deployit Server.
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param file The name of the package to import
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     */
    @POST
    @Path("import/{file:.*?}")
    ConfigurationItem importPackage(@PathParam("file") String file);

    /**
     * Fetches an {@link ImportablePackage application package} from a URL outside of the Deployit Server.
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param url The URL to import from
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     */
    @POST
    @Path("fetch")
    ConfigurationItem fetch(String url);

    /**
     * Uploads an {@link ImportablePackage application package} to the Deployit Server.
     *
     * @permission import#initial for importing the first version of a package
     * @permission import#upgrade if another version of the package already exists
     * @param file The name of the package to be imported
     * @param form The upload form.
     * @return The {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} CI that is created after importing.
     */
    @POST
    @Path("upload/{file:.*?}")
    @Consumes(MediaType.MULTIPART_FORM_DATA)
    ConfigurationItem upload(@PathParam("file") String file, @MultipartForm FileUpload form);
}
