package com.xebialabs.deployit.documentation;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.deployit.plugin.api.rules.StepParameter;

import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;

public class StepReferenceGenerator {

    private final Scannit scannit;

    private final StepReferenceFreemarkerTemplate template;

    private final List<File> markdownSources;
    private final ContextProperties contextProperties;

    public StepReferenceGenerator(final String packageName, Writer writer, final List<File> markdownSources, final ContextProperties contextProperties) {
        this.markdownSources = markdownSources;
        this.contextProperties = contextProperties;
        try {
            final Configuration configuration = Configuration.config().scan(packageName).with(new TypeAnnotationScanner());
            this.scannit = new Scannit(configuration);
            this.template = new StepReferenceFreemarkerTemplate(writer);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void generateStepReference() {
        try {
            final List<Step> steps = findSteps();
            if (!steps.isEmpty()) {
                final HashMap<String, Object> templateData = new HashMap<String, Object>();
                templateData.put("steps", steps);
                template.writeGeneratedPage(templateData);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private List<Step> findSteps() {
        return newArrayList(transform(scannit.getTypesAnnotatedWith(StepMetadata.class), new ClassToStep()));
    }

    private class ClassToStep implements Function<Class<?>, Step> {
        @Override
        public Step apply(final Class<?> stepClass) {
            final List<Parameter> parameters = newArrayList(transform(getAllFields(stepClass, StepParameter.class), new FiledToParameter()));
            final String name = stepClass.getAnnotation(StepMetadata.class).name();
            return new Step(name, generateStepDescription(name), parameters);
        }

        private List<Field> getAllFields(Class<?> aClass, Class<? extends Annotation> annotation) {
            List<Field> result = new ArrayList<Field>();

            for (Field field : aClass.getDeclaredFields()) {
                if (field.getAnnotation(annotation) != null) {
                    result.add(field);
                }
            }

            final Class<?> superclass = aClass.getSuperclass();
            if(superclass != null){
                result.addAll(getAllFields(superclass, annotation));
            }

            return result;
        }

        private String generateStepDescription(String stepName) {
            StringWriter writer = new StringWriter();
            final List<File> markdowns = Lists.newArrayList(Iterables.filter(markdownSources, new StepFilenameFilter(stepName)));
            for (File markdown : markdowns) {
                new MarkdownHtmlGenerator(markdown, contextProperties).generate(writer);
            }
            return writer.getBuffer().toString();
        }
    }

    private static class StepFilenameFilter implements Predicate<File> {

        private String stepName;

        private StepFilenameFilter(final String stepName) {
            this.stepName = stepName;
        }

        @Override
        public boolean apply(final File file) {
            return "step-".concat(stepName.concat(".markdown")).equals(file.getName());
        }
    }

    private static class FiledToParameter implements Function<Field, Parameter> {
        @Override
        public Parameter apply(final Field field) {
            StepParameter stepParameter = field.getAnnotation(StepParameter.class);
            return new Parameter(
                    field.getName(),
                    field.getType().getSimpleName(),
                    stepParameter.label(),
                    stepParameter.description(),
                    stepParameter.required(),
                    stepParameter.calculated());
        }
    }

    @SuppressWarnings("unused")
    public static class Step {
        private String name;
        private String htmlDescription;
        private List<Parameter> parameters;

        private Step(final String name, final String htmlDescription, final List<Parameter> parameters) {
            this.name = name;
            this.parameters = parameters;
            this.htmlDescription = htmlDescription;
        }

        public String getName() {
            return name;
        }

        public List<Parameter> getParameters() {
            return parameters;
        }

        public String getHtmlDescription() {
            return htmlDescription;
        }
    }

    @SuppressWarnings("unused")
    public static class Parameter {
        String name;
        String label;
        String description;
        Boolean required;
        Boolean calculated;
        String type;

        private Parameter(final String name, final String type, final String label, final String description, Boolean required, Boolean calculated) {
            this.name = name;
            this.label = label;
            this.description = description;
            this.type = type;
            this.required = required;
            this.calculated = calculated;
        }

        public String getName() {
            return name;
        }

        public String getLabel() {
            return label;
        }

        public String getDescription() {
            return description;
        }

        public Boolean getRequired() {
            return required;
        }

        public String getType() {
            return type;
        }

        public Boolean getCalculated() {
            return calculated;
        }
    }
}
