package com.xebialabs.deployit.documentation;

import java.io.*;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Joiner;

import com.xebialabs.overthere.RuntimeIOException;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.booter.local.utils.Closeables.closeQuietly;
import static com.xebialabs.deployit.documentation.IOUtils.*;

public class WkhtmlPdfGenerator {

    private URL coverHtml;

    private URL footerHtml;

    private URL headerHtml;

    private URL tocXsl;

    private String wkHtmlToPdfCommand = "wkhtmltopdf";

    private PrintStream processOutputStream = System.out;

    private ContextProperties properties = new ContextProperties();
    private File workingDir;
    private Map<URL, File> tempFileCache = newHashMap();

    public File generatePdfFromHtml(File html) {
        checkArgument(workingDir != null, "Working directory must be provided.");
        String config = extractFileNameWithoutExtension(html);
        File pdf = createFileNameWithNewExtension(html, workingDir, "pdf");

        logger.info("Creating pdf '{}' from from html source '{}'.", pdf, html);

        ContextProperties placeholderValues = preparePlaceholderValues(config);

        List<String> command = newArrayList();
        command.add(wkHtmlToPdfCommand);

        List<File> tempFiles = newArrayList();
        command.add("--disable-javascript");
        command.add("--image-dpi");
        command.add("1024");
        command.add("--margin-bottom");
        command.add("12");
        command.add("--print-media-type");

        tempFiles.add(processOption(command, "html", placeholderValues, headerHtml, "--header-html"));
        tempFiles.add(processOption(command, "html", placeholderValues, footerHtml, "--footer-html"));
        tempFiles.add(processOption(command, "html", placeholderValues, coverHtml, "cover"));
        tempFiles.add(processOption(command, "xsl", placeholderValues, tocXsl, "toc", "--xsl-style-sheet"));
        command.add(html.getAbsolutePath());
        command.add(pdf.getAbsolutePath());

        executeWkhtmlCommand(command);
        deleteTempFiles(tempFiles);
        return pdf;
    }

    private ContextProperties preparePlaceholderValues(String config) {
        ContextProperties values = new ContextProperties();
        values.putAll(properties);
        String docName = values.get(config + ".title", config);
        values.put("title", docName);
        return values;
    }

    private File processOption(List<String> cmd, String fileExt, ContextProperties values, URL url, String... options) {
        if (url != null) {
            File file = createFileForUrlWithPlaceholdersReplaced(url, values, fileExt);
            cmd.addAll(Arrays.asList(options));
            cmd.add(file.getAbsolutePath());
            return file;
        }
        return null;
    }

    private File createFileForUrlWithPlaceholdersReplaced(URL url, ContextProperties values, String fileExt) {
        try {
            File target = downloadUrl(url, fileExt);
            File placeholderReplacedTarget = createTempFile(fileExt);
            FileWriter fw = new FileWriter(placeholderReplacedTarget);
            FileReader fr = new FileReader(target);
            try {
                replacePlaceholders(fr, values, fw);
            } finally {
                closeQuietly(fw);
                closeQuietly(fr);
            }
            return placeholderReplacedTarget;
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private File downloadUrl(final URL url, String fileExt) throws IOException {
        if (tempFileCache.containsKey(url)) {
            return tempFileCache.get(url);
        }

        File tempFile = createTempFile(fileExt);
        copy(url, tempFile);
        tempFileCache.put(url, tempFile);

        return tempFile;
    }

    private static void deleteTempFiles(Collection<File> tempFiles) {
        for (File file : tempFiles) {
            if (file != null) {
                file.delete();
            }
        }
    }

    public void clearTemporaryFiles() {
        deleteTempFiles(tempFileCache.values());
    }

    private File createTempFile(String ext) throws IOException {
        return File.createTempFile("deployit", "." + ext, workingDir);
    }

    private void executeWkhtmlCommand(List<String> command) {
        try {
            int rc = executeCommand(command.toArray(new String[command.size()]));
            if (rc != 0) {
                throw new RuntimeException("Failed to generate pdf. Return code '" + rc + "' from executing : " + command);
            }
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private int executeCommand(String[] command) throws IOException {

        ProcessBuilder probuilder = new ProcessBuilder(command);
        probuilder.redirectErrorStream(true);
        probuilder.directory(workingDir);
        logger.info("Executing command {}", Joiner.on(' ').join(command));
        Process process = probuilder.start();
        BufferedReader br = new BufferedReader(new InputStreamReader(process.getInputStream()));
        String line;

        while ((line = br.readLine()) != null) {
            logger.debug(line);
            if (!line.startsWith("[")) {  //don't print out ascii progress bar
                processOutputStream.println(line);
            }
        }

        //Wait to get exit value
        try {
            int exitValue = process.waitFor();
            return exitValue;
        } catch (InterruptedException e) {
            Thread.interrupted();
            logger.error("Error while waiting for command", e);
            return 1;
        }
    }

    public void setCoverHtml(URL coverHtml) {
        this.coverHtml = coverHtml;
    }

    public void setFooterHtml(URL footerHtml) {
        this.footerHtml = footerHtml;
    }

    public void setHeaderHtml(URL headerHtml) {
        this.headerHtml = headerHtml;
    }

    public void setTocXsl(URL tocXsl) {
        this.tocXsl = tocXsl;
    }

    public void setWkHtmlToPdfCommand(String wkHtmlToPdfCommand) {
        this.wkHtmlToPdfCommand = wkHtmlToPdfCommand;
    }

    public void setProcessOutputStream(PrintStream processOutputStream) {
        this.processOutputStream = processOutputStream;
    }

    public void setWorkingDir(File workingDir) {
        this.workingDir = workingDir;
    }

    public void setProperties(ContextProperties properties) {
        this.properties = properties;
    }

    private static final Logger logger = LoggerFactory.getLogger(WkhtmlPdfGenerator.class);

}
