package com.xebialabs.xltype.serialization.xstream;

import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;

import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.*;
import static com.xebialabs.xltype.serialization.xstream.Converters.writeNode;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;


public abstract class AbstractTaskConverter implements Converter {

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        TaskState t = (TaskState) source;
        writer.addAttribute("id", t.getId());
        writer.addAttribute("currentStep", Integer.toString(t.getCurrentStepNr()));
        writer.addAttribute("totalSteps", Integer.toString(t.getNrSteps()));
        writer.addAttribute("failures", Integer.toString(t.getFailureCount()));

        // Cleanup with ticket DEPL-5451
        // Backwards compatibility for UI. Task Monitor uses this TaskState object, instead of the new TaskWithBlockState.
        // 'state' attr should be old state, 'state2' should be new state for now.
        writer.addAttribute("state", backwardsCompatibleState(t.getState()).name());
        writer.addAttribute("state2", t.getState().name());
        writer.addAttribute("owner", t.getOwner());

        writeNode("description", t.getDescription(), writer);

        if (t.getStartDate() != null) {
            writeNode("startDate", new DateTimeAdapter().marshal(t.getStartDate()), writer);
        }
        if (t.getCompletionDate() != null) {
            writeNode("completionDate", new DateTimeAdapter().marshal(t.getCompletionDate()), writer);
        }
        if (t.getScheduledDate() != null) {
            writeNode("scheduledDate", new DateTimeAdapter().marshal(t.getScheduledDate()), writer);
        }

        writer.startNode("currentSteps");
        if (t.getCurrentStepNrs() != null) {
            for (Integer current : t.getCurrentStepNrs()) {
                if (current != null) {
                    writeNode("current", current.toString(), writer);
                }
            }
        }
        writer.endNode();

        Map<String, String> metadata = t.getMetadata();
        writer.startNode("metadata");
        for (Map.Entry<String, String> e : metadata.entrySet()) {
            writeNode(e.getKey(), e.getValue(), writer);
        }
        writer.endNode();

        if (t instanceof TaskWithSteps) {
            writeSteps(writer, context, (TaskWithSteps) t);
        }

    }

    protected TaskExecutionState backwardsCompatibleState(TaskExecutionState t) {
        TaskExecutionState retval = t;
        if (t == SCHEDULED) {
            retval = QUEUED;
        } else if (EnumSet.of(STOPPING, FAILING, ABORTING).contains(t)) {
            retval = EXECUTING;
        } else if (EnumSet.of(ABORTED, FAILED).contains(t)) {
            retval = STOPPED;
        }
        return retval;
    }

    private void writeSteps(HierarchicalStreamWriter writer, MarshallingContext context, TaskWithSteps t) {
        writer.startNode("steps");
        Converters.writeSteps(t.getSteps(), writer, context);
        writer.endNode();
    }

    @Override
    public boolean canConvert(Class type) {
        return TaskState.class.isAssignableFrom(type);
    }
}
