/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import java.util.*;

import com.xebialabs.deployit.plugin.api.udm.ExternalProperty;
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.joda.time.DateTime;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.SerializationException;
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter;

public class CiJsonReader implements CiReader {

    private final JSONObject json;

    private Iterator<String> propertyIterator;
    private String currentPropertyName;

    public CiJsonReader(JSONObject json) {
        this.json = json;
        this.propertyIterator = properties(json).iterator();
    }

    public static CiJsonReader create(String jsonObject) {
        try {
            return new CiJsonReader(new JSONObject(jsonObject));
        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse the following as a JSON object:\n" + jsonObject, e);
        }
    }

    //
    // CiReader implementation
    //

    @Override
    public String getType() {
        try {
            return json.getString("type");
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getId() {
        try {
            return json.getString("id");
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public String getToken() {
        return getStringOrNull("$token");
    }

    @Override
    public CiAttributes getCiAttributes() {
        final String createdBy = getStringOrNull("$createdBy");
        final DateTime createdAt = getDateTimeOrNull("$createdAt");
        final String lastModifiedBy = getStringOrNull("$lastModifiedBy");
        final DateTime lastModifiedAt = getDateTimeOrNull("$lastModifiedAt");
        final Long scmTraceabilityDataId = getLongOrNull("$scmTraceabilityDataId");
        return new CiAttributes(createdBy, createdAt, lastModifiedBy, lastModifiedAt, scmTraceabilityDataId);
    }

    @Override
    public boolean hasMoreProperties() {
        return propertyIterator.hasNext();
    }

    @Override
    public void moveIntoProperty() {
        currentPropertyName = propertyIterator.next();
    }

    @Override
    public CiReader moveIntoNestedProperty() {
        try {
            return new CiJsonReader(json.getJSONObject(currentPropertyName));
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public void moveOutOfProperty() {
        // Noop
    }

    @Override
    public String getCurrentPropertyName() {
        return currentPropertyName;
    }

    @Override
    public String getStringValue() {
        try {
            return json.getString(currentPropertyName);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<String> getStringValues() {
        try {
            JSONArray strings = json.getJSONArray(currentPropertyName);
            return toList(strings);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public Map<String, String> getStringMap() {
        try {
            JSONObject map = json.getJSONObject(currentPropertyName);
            return toMap(map);
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public boolean isCiReference() {
        return currentPropertyName != null && json.optJSONObject(currentPropertyName) == null;
    }

    @Override
    public String getCiReference() {
        return getStringValue();
    }

    @Override
    public List<String> getCiReferences() {
        return getStringValues();
    }

    @Override
    public CiListReader getCurrentCiListReader() {
        try {
            return new CiListJsonReader(json.getJSONArray(currentPropertyName));
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public List<ValidationMessage> getValidationMessages() {
        try {
            List<ValidationMessage> messages = new ArrayList<>();
            JSONArray array = json.getJSONArray("validation-messages");
            for (int i = 0; i < array.length(); i++) {
                messages.add(toMessage(array.getJSONObject(i)));
            }
            return messages;
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    @Override
    public Map<String, ExternalProperty> getExternalProperties() {
        try {
            HashMap<String, ExternalProperty> result = new HashMap<>();
            JSONObject object = json.getJSONObject("external-properties");
            Iterator<?> keys = object.keys();
            while (keys.hasNext()) {
                String propertyName = keys.next().toString();
                JSONObject jsonObject = object.getJSONObject(propertyName);
                String kind = jsonObject.getString("kind");
                if (kind.equals("lookup")) {
                    LookupValueKey lookupValueKey = new LookupValueKey();
                    lookupValueKey.setKey(jsonObject.getString("key"));
                    lookupValueKey.setProviderId(jsonObject.getString("provider"));
                    result.put(propertyName, lookupValueKey);
                } else {
                    throw new IllegalArgumentException("Unknown external property kind: '" + kind +"'.");
                }
            }
            return result;
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
    }

    //
    // Util
    //

    private static Collection<String> properties(JSONObject object) {
        List<String> properties = new ArrayList<>();
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = keys.next().toString();
            if(!isSpecialKey(key)) {
                properties.add(key);
            }
        }

        return properties;
    }

    private static boolean isSpecialKey(String key) {
        return "id".equals(key) || "type".equals(key) || (key != null && key.startsWith("$"));
    }

    private static List<String> toList(JSONArray array) throws JSONException {
        List<String> strings = new ArrayList<>(array.length());
        for (int i = 0; i < array.length(); i++) {
            strings.add(array.getString(i));
        }

        return strings;
    }

    private static Map<String, String> toMap(JSONObject object) throws JSONException {
        Map<String, String> map = new LinkedHashMap<>();
        Iterator<?> keys = object.keys();
        while (keys.hasNext()) {
            String key = keys.next().toString();
            map.put(key, object.getString(key));
        }

        return map;
    }

    private static ValidationMessage toMessage(JSONObject jsonObject) throws JSONException {
        return new ValidationMessage(
                jsonObject.getString("ci"),
                getStringOrNull(jsonObject, "property"),
                jsonObject.getString("message"),
                getStringOrNull(jsonObject, "level"));
    }

    private String getStringOrNull(String key) {
        return getStringOrNull(json, key);
    }

    private static String getStringOrNull(JSONObject json, String key) {
        try {
            if (json.has(key)) {
                return json.getString(key);
            }
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
        return null;
    }

    private DateTime getDateTimeOrNull(String key) {
        String stringOrNull = getStringOrNull(key);
        if (stringOrNull == null) {
            return null;
        }
        return new DateTimeAdapter().unmarshal(stringOrNull);
    }

    private Long getLongOrNull(String key) {
        return getLongOrNull(json, key);
    }

    private static Long getLongOrNull(JSONObject json, String key) {
        try {
            if (json.has(key)) {
                return json.getLong(key);
            }
        } catch (JSONException e) {
            throw new SerializationException(e);
        }
        return null;
    }
}
