package com.xebialabs.deployit.booter.local.utils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.concurrent.ExecutionException;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

public class ReflectionUtils {

    private static class ClassFieldName {
        private Class<?> clazz;
        private String fieldName;

        public ClassFieldName(Class<?> clazz, String fieldName) {
            this.clazz = clazz;
            this.fieldName = fieldName;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            final ClassFieldName that = (ClassFieldName) o;

            return clazz.equals(that.clazz) && fieldName.equals(that.fieldName);
        }

        @Override
        public int hashCode() {
            int result = clazz.hashCode();
            result = 31 * result + fieldName.hashCode();
            return result;
        }
    }

    private static final LoadingCache<ClassFieldName, Field> FIELD_CACHE = CacheBuilder.newBuilder().build(new CacheLoader<ClassFieldName, Field>() {
        @Override
        public Field load(ClassFieldName key) throws Exception {
            return _searchField(key.clazz, key.fieldName);
        }
    });


    public static Field searchField(Class<?> clazz, String name) {
        try {
            return FIELD_CACHE.get(new ClassFieldName(clazz, name));
        } catch (ExecutionException e) {
            throw new IllegalArgumentException(e);
        }
    }
    public static Field _searchField(Class<?> clazz, String name) {
        for (Class<?> scan = clazz; !scan.equals(Object.class); scan = scan.getSuperclass()) {
            try {
                Field declaredField = scan.getDeclaredField(name);
                declaredField.setAccessible(true);
                return declaredField;
            } catch (NoSuchFieldException e) {
                // scan up the tree
            }
        }
        throw new IllegalArgumentException("Cannot find '" + name + "' field on " + clazz.getName());
    }

    public static RuntimeException handleInvocationTargetException(InvocationTargetException e, String msg) {
        Throwable cause = e.getCause();
        if (cause != null) {
            return new RuntimeException(msg + "\n" + cause.getMessage(), cause);
        } else {
            return new RuntimeException(msg, e);
        }
    }

    public static void setField(Object on, String fieldName, Object value) {
        Field field = searchField(on.getClass(), fieldName);
        setField(on, field, value);
    }

    public static Object getField(Object on, Field field) {
        try {
            return field.get(on);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(format("Could not get field %s on %s.", field.getName(), on), e);
        }
    }

    static void setField(Object on, Field field, Object value) {
        try {
            field.set(on, value);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(format("Could not set field %s on %s to value %s.", field.getName(), on, value), e);
        }
    }

    public static List<Class<?>> getAllInterfaces(Class<?> cls) {
        if (cls == null) {
            return null;
        }

        List<Class<?>> interfacesFound = newArrayList();
        getAllInterfaces(cls, interfacesFound);

        return interfacesFound;
    }

    private static void getAllInterfaces(Class<?> cls, List<Class<?>> interfacesFound) {
        while (cls != null) {
            Class<?>[] interfaces = cls.getInterfaces();

            for (Class<?> anInterface : interfaces) {
                if (!interfacesFound.contains(anInterface)) {
                    interfacesFound.add(anInterface);
                    getAllInterfaces(anInterface, interfacesFound);
                }
            }

            cls = cls.getSuperclass();
        }
    }

    public static void setFieldWithConversion(final Object on, final String fieldName, final String value) {
        Field field = searchField(on.getClass(), fieldName);
        Class<?> type = field.getType();
        try {
            if (type.equals(String.class)) {
                setField(on, field, value);
            } else if (type.equals(int.class) || type.equals(Integer.class)) {
                field.set(on, Integer.parseInt(value));
            } else if (type.equals(boolean.class) || type.equals(Boolean.class)) {
                field.set(on, Boolean.parseBoolean(value));
            } else {
                throw new IllegalArgumentException(format("Cannot yet convert type [%s] for field [%s]", type, field));
            }
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(format("Could not set field %s on %s to value %s.", field.getName(), on, value), e);
        }
    }
}
