package com.xebialabs.deployit.booter.local;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.MethodAnnotationScanner;
import nl.javadude.scannit.scanner.SubTypeScanner;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;

/**
 * Class responsible for booting all the plugins. Responsible for: - Registering all the Descriptors - Setting up the
 * Global Context.
 */
public class LocalBooter {

    private static final AtomicBoolean isBooted = new AtomicBoolean(false);
    private static final AtomicBoolean initialized = new AtomicBoolean(false);

    /**
     * Boot the XL Deploy Plugin System. Setting everything up.
     */
    public static void boot() {
        try {
            boot(true, Collections.<String> emptyList());
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System. Setting everything up. You can also specify extra packages to scan for classes
     * with XL Deploy annotations.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void boot(List<String> packagesToScan) {
        try {
            boot(true, packagesToScan);
        } catch (RuntimeException re) {
            logger.error("Quitting, could not boot plugins: ", re);
            throw re;
        }
    }

    /**
     * Boot the XL Deploy Plugin System, but without initializing the global context with user overridden default
     * values. Useful for tests where you want to have control which default values a CI has.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void bootWithoutGlobalContext() {
        boot(false, Collections.<String> emptyList());
    }

    /**
     * Boot the XL Deploy Plugin System, but without initializing the global context with user overridden default
     * values. You can also specify extra packages to scan for classes with XL Deploy annotations. Useful for tests
     * where you want to have control which default values a CI has.
     * <p>
     * Note: The {@link LocalBooter} is a singleton, changing the packages to scan after it has been instantiated has no
     * effect.
     */
    public static void bootWithoutGlobalContext(List<String> packagesToScan) {
        boot(false, packagesToScan);
    }

    private synchronized static void boot(boolean withGlobalContext, List<String> packagesToScan) {
        if (!isBooted.getAndSet(true)) {
            LocalDescriptorRegistry registry = new LocalDescriptorRegistry();
            DescriptorRegistry.add(registry);

            Configuration configuration = Configuration.config()
                .scan("com.xebialabs")
                .scan("ext.deployit") // Deployit Extensions
                .with(new TypeAnnotationScanner(), new MethodAnnotationScanner(), new SubTypeScanner());

            for (String pkg : packagesToScan) {
                logger.debug("Scanning additional package '{}' for types, rules etc.", pkg);
                configuration.scan(pkg);
            }

            Scannit.boot(configuration);
            DelegateRegistry.boot();
            TypeDefinitions typeDefinitions = new TypeDefinitions(registry);
            TypeSystemBootstrapper typeSystemBootstrapper = new TypeSystemBootstrapper();
            typeSystemBootstrapper.bootstrap(typeDefinitions);
            Verifications verifications = new Verifications();
            DelegateRegistry.verify(verifications);
            typeSystemBootstrapper.verifyTypes(verifications);
            verifications.done();
            if (withGlobalContext) {
                GlobalContextInitializer.init();
            }
            PluginVersions.init();
            initialized.set(true);
        }

        if (!initialized.get()) {
            throw new IllegalStateException("The DescriptorRegistry has been booted, but is not initialized. Please check the logs for any errors.");
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(LocalBooter.class);
}
