package com.xebialabs.deployit.booter.local;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

public class LocalDescriptorRegistry extends DescriptorRegistry {
    public static final DescriptorRegistryId LOCAL_ID = new LocalDescriptorRegistryId();

    private final Map<Type, Descriptor> descriptors = Maps.newHashMap();

    private final Multimap<Type, Type> subtypes = HashMultimap.create();

    // N.B.: Logger must be defined at top of file so that other static initializers can access it
    private static Logger logger = LoggerFactory.getLogger(LocalDescriptorRegistry.class);

    protected LocalDescriptorRegistry() {
        super(LOCAL_ID);
    }

    void register(Descriptor descriptor) {
        checkState(!this.descriptors.containsKey(descriptor.getType()), "The type [%s] is already registered in XL Deploy.", descriptor.getType());
        logger.debug("Registered ConfigurationItem: {}", descriptor);

        this.descriptors.put(descriptor.getType(), descriptor);
    }

    void registerSubtype(Type supertype, Type subtype) {
        checkState(!supertype.equals(subtype), "Cannot register [%s] as its own subtype.", supertype);
        this.subtypes.put(supertype, subtype);
    }

    @Override
    protected boolean isLocal() {
        return true;
    }

    @Override
    protected Collection<Descriptor> _getDescriptors() {
        return Collections.unmodifiableCollection(descriptors.values());
    }

    @Override
    protected Collection<Type> _getSubtypes(Type supertype) {
        return Collections.unmodifiableCollection(subtypes.get(supertype));
    }

    @Override
    protected Descriptor _getDescriptor(Type type) {
        checkState(!descriptors.isEmpty(), "DescriptorRegistry not booted");
        checkArgument(exists(type), "Unknown type [%s]", type);
        return descriptors.get(type);
    }

    @Override
    protected boolean _exists(Type type) {
        return descriptors.containsKey(type);
    }
}
