package com.xebialabs.deployit.booter.local;

import com.google.common.collect.Iterators;
import com.xebialabs.deployit.booter.local.generator.TypeGenerators;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import nl.javadude.scannit.Scannit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.io.IOException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Predicates.equalTo;
import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.booter.local.SyntheticHelper.readSyntheticDocument;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.childByName;
import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;

class TypeSystemBootstrapper {

    private final TypeGenerators typeGenerators;

    TypeSystemBootstrapper(TypeGenerators typeGenerators) {
        this.typeGenerators = typeGenerators;
    }

    void bootstrap(TypeDefinitions typeDefs) {
        logger.info("Bootstrapping the Type System.");
        scanClasses(typeDefs);
        scanSynthetics(typeDefs);
        createTypeTree(typeDefs);
        registerTypeDefinitions(typeDefs);
        typeGenerators.generateTypeDefinitions(typeDefs);
        createTypeTree(typeDefs);
        registerTypeDefinitions(typeDefs);
    }

    private void createTypeTree(TypeDefinitions typeDefs) {
        typeDefs.getDefinitions().forEach(TypeDefinitions.TypeDefinition::registerTypeTree);
    }

    void verifyTypes(Verifications verifications) {
        for (Descriptor descriptor : from(DescriptorRegistry.getDescriptors()).filter(input -> !input.isVirtual())) {
            ((LocalDescriptor) descriptor).verify(verifications);
        }
    }

    private void scanClasses(TypeDefinitions typeDefs) {
        typeDefs.defineType(ConfigurationItem.class);
        Set<Class<? extends ConfigurationItem>> cis = Scannit.getInstance().getSubTypesOf(ConfigurationItem.class);
        for (Class<? extends ConfigurationItem> ci : cis) {
            typeDefs.defineType(ci);
        }
    }

    private void scanSynthetics(TypeDefinitions typeDefs) {
        try {
            List<Element> types = newArrayList();
            List<Element> typeModifications = newArrayList();
            readSynthetics("synthetic.xml", types, typeModifications);
            readSynthetics("synthetic-test.xml", types, typeModifications);

            parseAllTypeDefinitions(types, typeDefs);
            parseAllTypeModifications(typeDefs, typeModifications);
        } catch (IOException ex) {
            throw new RuntimeException("Could not read synthetic type definitions.", ex);
        }
    }

    private void registerTypeDefinitions(TypeDefinitions typeDefs) {
        for (TypeDefinitions.TypeDefinition typeDefinition : typeDefs.getDefinitions()) {
            typeDefinition.register(typeDefs);
        }
    }

    private void parseAllTypeDefinitions(List<Element> types, TypeDefinitions typeDefs) {
        for (Element type : types) {
            typeDefs.defineType(type);
            Iterator<Element> generatedTypes = childByName(type, equalTo("generate-deployable"));
            if (generatedTypes.hasNext()) {
                typeDefs.defineGeneratedType(generatedTypes.next(), type);
            }

            findAllGeneratedParameterTypes(typeDefs, type);
        }
    }

    private void findAllGeneratedParameterTypes(final TypeDefinitions typeDefs, final Element type) {
        Iterator<Element> methodDefs = childByName(type, equalTo("method"));
        while (methodDefs.hasNext()) {
            Element methodDef = methodDefs.next();
            Iterator<Element> parameters = childByName(methodDef, equalTo("parameters"));
            if (parameters.hasNext()) {
                typeDefs.defineGeneratedParameterType(methodDef, type);
            }
        }
    }

    private void parseAllTypeModifications(TypeDefinitions typeDefs, List<Element> typeModifications) {
        for (Element typeModification : typeModifications) {
            typeDefs.modifyType(typeModification);
            findAllGeneratedParameterTypes(typeDefs, typeModification);
        }
    }

    private void readSynthetics(String name, List<Element> typeDefinitionCollector, List<Element> typeModificationsCollector) throws IOException {
        Enumeration<URL> syntheticXMLs = MODULE$.classLoader().getResources(name);
        while (syntheticXMLs.hasMoreElements()) {
            final URL syntheticXML = syntheticXMLs.nextElement();
            logger.debug("Scanning synthetic XML: {}", syntheticXML);
            Element docElement = readSyntheticDocument(syntheticXML).getDocumentElement();

            Iterators.addAll(typeDefinitionCollector, childByName(docElement, equalTo("type")));
            Iterators.addAll(typeModificationsCollector, childByName(docElement, equalTo("type-modification")));
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeSystemBootstrapper.class);
}
