package com.xebialabs.deployit.booter.local;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Splitter;
import com.google.common.collect.*;
import com.xebialabs.deployit.booter.local.validation.CollectionTypeValidator;
import com.xebialabs.deployit.booter.local.validation.ReferenceCollectionTypeValidator;
import com.xebialabs.deployit.booter.local.validation.ReferenceTypeValidator;
import com.xebialabs.deployit.booter.local.validation.RequiredValidator;
import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.CandidateValuesFilter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.ValidationContext;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;
import nl.javadude.scannit.Scannit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.bind.DatatypeConverter;
import java.lang.reflect.Method;
import java.util.*;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static java.lang.String.format;
import static java.util.EnumSet.of;

public abstract class LocalPropertyDescriptor implements PropertyDescriptor {

    protected final Logger logger = LoggerFactory.getLogger(getClass());

    private LocalDescriptor declaringDescriptor;
    private String fullyQualifiedName;
    private String name;

    private boolean asContainment;
    private String category;
    private String description;
    private String label;
    private boolean password;
    private boolean required;
    private Property.Size size;
    private PropertyKind kind;
    private List<String> enumValues;

    private Class<?> enumClass;
    private Type referencedType;
    private boolean hidden;
    private boolean inspectionProperty;
    private boolean requiredForInspection;
    private boolean isTransient;
    protected Set<Validator<?>> validationRules = newHashSet();
    private Set<String> aliases = newHashSet();
    private String candidateValuesFilter;
    private boolean deployedSpecific;

    protected void reInitializeRequired() {
        if (kind == PropertyKind.BOOLEAN && required) {
            logger.warn("Required boolean property [{}] will be treated as an optional property.", this);
            required = false;
        }

        if ((kind == PropertyKind.LIST_OF_CI || kind == PropertyKind.SET_OF_CI) && asContainment && required) {
            logger.warn("Required asContainment collection of configuration items [{}] will be treated as an optional property.", this);
            required = false;
        }
    }

    LocalPropertyDescriptor copyWithNewDescriptor(LocalDescriptor declaringDescriptor) {
        return new LocalPropertyDescriptorWithDifferentOwner(this, declaringDescriptor);
    }

    protected void setFromPropertyDescriptor(LocalPropertyDescriptor pd) {
        this.setName(pd.getName());
        this.setAsContainment(pd.isAsContainment());
        this.setCategory(pd.getCategory());
        this.setDescription(pd.getDescription());
        this.setLabel(pd.getLabel());
        this.setPassword(pd.isPassword());
        this.setRequired(pd.isRequired());
        this.setSize(pd.getSize());
        this.setKind(pd.getKind());
        this.setEnumValues(pd.getEnumValues());
        this.setEnumClass(pd.getEnumClass());
        this.setReferencedType(pd.getReferencedType());
        this.setHidden(pd.isHidden());
        this.setInspectionProperty(pd.isInspectionProperty());
        this.setRequiredForInspection(pd.isRequiredForInspection());
        this.setValidationRules(newHashSet(pd.validationRules));
        this.setAliases(newHashSet(pd.getAliases()));
        this.setTransient(pd.isTransient());
        this.setCandidateValuesFilter(pd.getCandidateValuesFilter());
        this.setDeployedSpecific(pd.isDeployedSpecific());
    }

    public void verify(Verifications verifications) {
        verifyName(verifications, "name");
        // UI naming convention.
        verifyName(verifications, "displayName");
        verifyName(verifications, "id");
        verifyName(verifications, "type");
        verifications.verify(!getName().contains("$"), "Cannot define a property named '%s' because it contains a '$'", getName());
        verifications.verify(!hidden || isTransient, "Hidden property '%s' should be transient", this);

        boolean isCiKind = of(LIST_OF_CI, SET_OF_CI, CI).contains(kind);
        verifications.verify(!asContainment || isCiKind, "'%s' can only be an as containment relation if it is a 'set_of_ci', 'list_of_ci' or 'ci' kind property", this);
        verifications.verify(referencedType == null || isCiKind, "'%s' can only have reference type '%s' set when it is a (Set/List of) CI kind property", this, referencedType);
        verifications.verify(candidateValuesFilter == null || isCiKind, "'%s' can only define a candidate-values-filter if it is a 'set_of_ci', 'list_of_ci' or 'ci' kind property", this);
        verifications.verify(candidateValuesFilter == null || findFilterMethod() != null, "'%s' refers to an unknown candidate-values-filter '%s'", this, candidateValuesFilter);
        verifications.verify(!asContainment || isCiKind, "'%s' can only be an as containment relation if it is a 'set_of_ci', 'list_of_ci' or 'ci' kind property", this);
        verifications.verify(referencedType == null || isCiKind, "'%s' can only have reference type '%s' set when it is a (Set/List of) CI kind property", this, referencedType);
        boolean isPasswordKind = of(STRING, SET_OF_STRING, LIST_OF_STRING, MAP_STRING_STRING).contains(kind);
        if (password && !isPasswordKind) {
            Deprecations.deprecated("'%s' can only be a password property if it is a 'string', 'set_of_string', 'list_of_string' or 'map_string_string' kind property", this);
        }

        verifyAliases(verifications);
    }

    private void verifyAliases(Verifications verifications) {
        if (!aliases.isEmpty()) {
            for (PropertyDescriptor otherPd : declaringDescriptor.getPropertyDescriptors()) {
                LocalPropertyDescriptor pd = (LocalPropertyDescriptor) otherPd;
                if (!otherPd.equals(this)) {
                    verifications.verify(declaringDescriptor.getType(), !aliases.contains(pd.getName()), "Aliases of [%s] contain name [%s] which is an existing property.", this, pd.getName());
                    ImmutableSet<String> intersection = Sets.intersection(aliases, pd.aliases).immutableCopy();
                    verifications.verify(declaringDescriptor.getType(), intersection.isEmpty(), "Aliases of [%s] conflict with aliases of [%s]. Conflicting: %s", this, pd, intersection);
                }
            }
        }
    }

    private void verifyName(Verifications verifications, String name) {
        verifications.verify(!this.name.equals(name), "Cannot define a property named '%s' on %s", name, declaringDescriptor);
    }


    private Method findFilterMethod() {
        Set<Method> filters = Scannit.getInstance().getMethodsAnnotatedWith(CandidateValuesFilter.class);
        return Iterables.tryFind(filters, input -> input.getAnnotation(CandidateValuesFilter.class).name().equals(candidateValuesFilter)).orNull();
    }

    protected void initEnumValues(Class<?> enumClass) {
        enumValues = newArrayList();
        this.enumClass = enumClass;
        for (Enum<?> enumValue : (Enum<?>[]) enumClass.getEnumConstants()) {
            enumValues.add(enumValue.name());
        }
    }

    protected void addDefaultValidationRules() {
        if (required && !asContainment && kind != PropertyKind.BOOLEAN) {
            validationRules.add(new RequiredValidator());
        }

        if (of(SET_OF_CI, SET_OF_STRING, LIST_OF_CI, LIST_OF_STRING).contains(kind)) {
            validationRules.add(new CollectionTypeValidator(this));
        }

        if (of(SET_OF_CI, LIST_OF_CI).contains(kind)) {
            validationRules.add(new ReferenceCollectionTypeValidator(this));
        } else if (CI == kind) {
            validationRules.add(new ReferenceTypeValidator(this));
        }
    }


    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public boolean isAsContainment() {
        return asContainment;
    }

    @Override
    public String getCategory() {
        return category;
    }

    @Override
    public String getLabel() {
        return label;
    }

    @Override
    public boolean isPassword() {
        return password;
    }

    @Override
    public boolean isRequired() {
        return required;
    }

    @Override
    public Property.Size getSize() {
        return size;
    }

    @Override
    public PropertyKind getKind() {
        return kind;
    }

    @Override
    public List<String> getEnumValues() {
        return enumValues != null ? newArrayList(enumValues) : enumValues;
    }

    public Class<?> getEnumClass() {
        return enumClass;
    }

    public LocalDescriptor getDeclaringDescriptor() {
        return declaringDescriptor;
    }

    @Override
    public Type getReferencedType() {
        return referencedType;
    }

    @Override
    public Object getDefaultValue() {
        return convertValue(GlobalContext.lookup(this));
    }

    private Object convertValue(final String val) {
        if (val == null) return null;
        switch (kind) {
            case BOOLEAN:
                return Boolean.parseBoolean(val);
            case INTEGER:
                if (val.isEmpty()) return null;
                return Integer.parseInt(val);
            case STRING:
                return val;
            case ENUM:
                if (enumClass != null) {
                    for (Enum<?> enumConstant : (Enum<?>[]) enumClass.getEnumConstants()) {
                        if (enumConstant.name().equalsIgnoreCase(val)) {
                            return enumConstant;
                        }
                    }
                    throw new IllegalArgumentException("Value " + val + " not a member of enum " + enumClass);
                } else {
                    Optional<String> optionalValue = Iterables.tryFind(enumValues, input -> input.equalsIgnoreCase(val));
                    if (optionalValue.isPresent()) {
                        return optionalValue.get();
                    } else {
                        throw new IllegalArgumentException("Value " + val + " not a member of possible enum values");
                    }
                }
            case DATE:
                return DatatypeConverter.parseDateTime(val).getTime();
            case SET_OF_STRING:
                return newLinkedHashSet(splitValue(val));
            case LIST_OF_STRING:
                return newArrayList(splitValue(val));
            case MAP_STRING_STRING:
                return decodeMap(val);
            default:
                throw new IllegalArgumentException("Property " + name + " of kind " + kind + " cannot be converted from a string value");
        }
    }

    private Map<String, String> decodeMap(String val) {
        Map<String, String> map = newLinkedHashMap();
        for (String s : splitValue(val)) {
            String[] split = s.split(":");
            checkArgument(emptyToNull(split[0]) != null, "Property %s of kind %s cannot be converted from string value '%s' because of any empty key.", name, kind, val);
            if (split.length == 1) {
                map.put(split[0], "");
            } else {
                map.put(split[0], split[1]);
            }
        }
        return map;
    }

    private Iterable<String> splitValue(String val) {
        return Splitter.on(',').trimResults().omitEmptyStrings().split(val);
    }

    @Override
    public Set<String> getAliases() {
        return aliases;
    }

    @Override
    public String getCandidateValuesFilter() {
        return candidateValuesFilter;
    }

    @Override
    public boolean isHidden() {
        return hidden;
    }

    @Override
    public final void set(ConfigurationItem item, Object value) {
        if (value instanceof String) {
            value = convertValue((String) value);
        } else if (value == null) {
            value = getDefaultValue();
        }
        if (value == null) {
            value = emptyValue();
        }
        logger.trace("Setting value [{}] on property [{}] of CI [{}]", value, getFqn(), item.getId());
        doSetValue(item, value);
    }

    protected abstract void doSetValue(ConfigurationItem item, Object value);

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqual(item, other, new HashSet<>());
    }

    boolean areEqual(ConfigurationItem item, ConfigurationItem other, Set<String> itemsBeingCompared) {
        Object left = get(item);
        Object right = get(other);
        logger.trace("Comparing {}: old [{}] <-> new [{}]", getFqn(), left, right);
        return areValuesEqual(left, right, itemsBeingCompared);
    }

    @SuppressWarnings("unchecked")
    private boolean areValuesEqual(Object itemValue, Object otherValue, Set<String> itemsBeingCompared) {

        if (itemValue == null) {
            return otherValue == null;
        } else if (otherValue == null) {
            return false;
        }

        switch (kind) {
            case SET_OF_STRING:
                return symmetricDifference((Set<String>) itemValue, (Set<String>) otherValue).isEmpty();
            case SET_OF_CI:
                Function<ConfigurationItem, String> f = ConfigurationItem::getName;

                Iterable<ConfigurationItem> cis = Iterables.concat((Set<ConfigurationItem>) itemValue, (Set<ConfigurationItem>) otherValue);
                Multimap<String, ConfigurationItem> index = Multimaps.index(cis, f);
                for (String key : index.keySet()) {
                    Collection<ConfigurationItem> cisToCompare = index.get(key);
                    if (cisToCompare.size() != 2) {
                        return false;
                    }
                    Iterator<ConfigurationItem> itemIterator = cisToCompare.iterator();
                    ConfigurationItem lhs = itemIterator.next();
                    ConfigurationItem rhs = itemIterator.next();
                    if (areCiEqual(lhs, rhs, itemsBeingCompared)) continue;
                    return false;
                }
                return true;
            case CI:
                ConfigurationItem itemValueAsCi = (ConfigurationItem) itemValue;
                ConfigurationItem otherValueAsCi = (ConfigurationItem) otherValue;
                if (itemValueAsCi.getName().equals(otherValueAsCi.getName())) {
                    LocalDescriptor descriptor = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(itemValueAsCi.getType());
                    return descriptor.areEqualDeeply(itemValueAsCi, otherValueAsCi, itemsBeingCompared);
                }
                return false;
            case MAP_STRING_STRING:
                Map<String, String> left = (Map<String, String>) itemValue;
                Map<String, String> right = (Map<String, String>) otherValue;
                return Maps.difference(left, right).areEqual();
            case LIST_OF_STRING:
                List<String> leftStrings = (List<String>) itemValue;
                List<String> rightStrings = (List<String>) otherValue;
                // Don't compare the lists using equals, because we might be dealing with different kinds of lists.
                return Iterables.elementsEqual(leftStrings, rightStrings);
            case LIST_OF_CI:
                List<ConfigurationItem> lhs = (List<ConfigurationItem>) itemValue;
                List<ConfigurationItem> rhs = (List<ConfigurationItem>) otherValue;
                if (lhs.size() != rhs.size()) {
                    return false;
                } else {
                    Iterator<ConfigurationItem> lIter = lhs.iterator();
                    Iterator<ConfigurationItem> rIter = rhs.iterator();
                    ConfigurationItem lItem;
                    ConfigurationItem rItem;
                    while (lIter.hasNext()) {
                        lItem = lIter.next();
                        rItem = rIter.next();
                        if (areCiEqual(lItem, rItem, itemsBeingCompared)) continue;
                        return false;
                    }
                    return true;
                }
            default:
                return itemValue.equals(otherValue);
        }
    }

    private Set<String> symmetricDifference(Set<String> leftSet, Set<String> rightSet) {
        Set<String> symmetricDiff = new HashSet<>(leftSet);
        symmetricDiff.addAll(rightSet);
        Set<String> tmp = new HashSet<>(leftSet);
        tmp.retainAll(rightSet);
        symmetricDiff.removeAll(tmp);
        return symmetricDiff;
    }

    private boolean areCiEqual(ConfigurationItem lItem, ConfigurationItem rItem, Set<String> itemsBeingCompared) {
        if (lItem.getName().equals(rItem.getName()) && lItem.getType().equals(rItem.getType())) {
            LocalDescriptor descriptor = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(lItem.getType());
            if (descriptor.areEqualDeeply(lItem, rItem, itemsBeingCompared)) {
                return true;
            }
        }
        return false;
    }

    @SuppressWarnings("unchecked")
    void validate(final ConfigurationItem ci, final List<ValidationMessage> messages) {
        ValidationContext context = (message, params) -> messages.add(new ValidationMessage(ci.getId(), name, format(message, params)));
        for (Validator<?> validationRule : validationRules) {
            ((Validator<Object>) validationRule).validate(get(ci), context);
        }
    }

    @Override
    public boolean isInspectionProperty() {
        return inspectionProperty;
    }

    @Override
    public boolean isRequiredForInspection() {
        return requiredForInspection;
    }

    @Override
    public boolean isTransient() {
        return isTransient;
    }

    @Override
    public String getFqn() {
        if (fullyQualifiedName == null) {
            assignFullyQualifiedName();
        }
        return fullyQualifiedName;
    }

    private void assignFullyQualifiedName() {
        this.fullyQualifiedName = declaringDescriptor.getType() + "." + name;
    }

    public boolean isDeployedSpecific() {
        return deployedSpecific;
    }

    @Override
    public String toString() {
        return getFqn();
    }

    public Object emptyValue() {
        switch (kind) {
            case BOOLEAN:
                return false;
            case SET_OF_STRING:
            case SET_OF_CI:
                return newHashSet();
            case LIST_OF_STRING:
            case LIST_OF_CI:
                return newArrayList();
            case MAP_STRING_STRING:
                return newHashMap();
            default:
                return null;
        }
    }

    protected void setName(String name) {
        this.name = name;
    }

    protected void setAsContainment(boolean asContainment) {
        this.asContainment = asContainment;
    }

    protected void setCategory(String category) {
        this.category = category;
    }

    protected void setDescription(String description) {
        this.description = description;
    }

    protected void setLabel(String label) {
        this.label = label;
    }

    protected void setPassword(boolean password) {
        this.password = password;
    }

    protected void setRequired(boolean required) {
        this.required = required;
    }

    protected void setSize(Property.Size size) {
        this.size = size;
    }

    protected void setKind(PropertyKind kind) {
        this.kind = kind;
    }

    protected void setEnumValues(List<String> enumValues) {
        this.enumValues = enumValues;
    }

    protected void setEnumClass(Class<?> enumClass) {
        this.enumClass = enumClass;
    }

    protected void setHidden(boolean hidden) {
        this.hidden = hidden;
    }

    protected void setReferencedType(Type referencedType) {
        this.referencedType = referencedType;
    }

    protected void setInspectionProperty(boolean inspectionProperty) {
        this.inspectionProperty = inspectionProperty;
    }

    protected void setTransient(boolean aTransient) {
        isTransient = aTransient;
    }

    protected void setCandidateValuesFilter(String candidateValuesFilter) {
        this.candidateValuesFilter = candidateValuesFilter;
    }

    protected void setDeployedSpecific(boolean deployedSpecific) {
        this.deployedSpecific = deployedSpecific;
    }

    protected void setDeclaringDescriptor(LocalDescriptor declaringDescriptor) {
        this.declaringDescriptor = declaringDescriptor;
    }

    protected void setRequiredForInspection(boolean requiredForInspection) {
        this.requiredForInspection = requiredForInspection;
    }

    protected void setAliases(Set<String> aliases) {
        this.aliases = aliases;
    }

    protected void setValidationRules(Set<Validator<?>> validationRules) {
        this.validationRules = validationRules;
    }

    protected void registerDefault(String defaultValue) {
        GlobalContext.register(this, defaultValue);
    }

    protected void registerDefault(PropertyDescriptor superPropertyDescriptor) {
        GlobalContext.register(this, superPropertyDescriptor);
    }
}
