package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.TypeHint;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.ENUM;
import static java.lang.String.format;

class GeneratedDeployablePropertyDescriptor extends LocalPropertyDescriptor {

    public GeneratedDeployablePropertyDescriptor(LocalDescriptor descriptor, PropertyDescriptor deployedPropertyDescriptor) {
        setName(deployedPropertyDescriptor.getName());
        setDeclaringDescriptor(descriptor);
        initMetadataFromDescriptor(deployedPropertyDescriptor);
    }

    private void initMetadataFromDescriptor(PropertyDescriptor propertyDescriptor) {
        // Generated 'simple' Deployable properties always are of kind String so that placeholder/dictionaries values can be filled.
        setKind(propertyDescriptor.getKind().isSimple() ? PropertyKind.STRING : propertyDescriptor.getKind());
        setCategory(propertyDescriptor.getCategory());
        setLabel(propertyDescriptor.getLabel());
        setRequired(false);
        setPassword(propertyDescriptor.isPassword());
        setSize(propertyDescriptor.getSize());
        initDescription(propertyDescriptor);
        initTypeHint(propertyDescriptor);
    }

    private void initDescription(PropertyDescriptor propertyDescriptor) {
        String description = format("%s (%s)", propertyDescriptor.getDescription(), propertyDescriptor.getKind().name().toLowerCase());
        if (propertyDescriptor.getKind() == ENUM) {
            description += format(" values(%s)", String.join(", ", propertyDescriptor.getEnumValues()));
        }
        if (propertyDescriptor.getKind().isSimple() && propertyDescriptor.getDefaultValue() != null) {
            description += format(" default(%s)", propertyDescriptor.getDefaultValue());
        }
        setDescription(description);
    }

    private void initTypeHint(final PropertyDescriptor propertyDescriptor) {
        addTypeHintForRequiredProperty(propertyDescriptor);
        addTypeHintFromEnumProperty(propertyDescriptor);
        addTypeHintFromPropertyValidationRules(propertyDescriptor);
        copyTypeHintFromDeployedTypeHint(propertyDescriptor);
    }

    private TypeHint getOrCreateTypeHint(final PropertyKind kind) {
        TypeHint typeHint = getTypeHint() != null ? getTypeHint() : new TypeHint();
        typeHint.setKind(kind);
        return typeHint;
    }

    private void addTypeHintForRequiredProperty(PropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor.isRequired()) {
            TypeHint typeHint = getOrCreateTypeHint(propertyDescriptor.getKind());
            typeHint.setRequired(true);
            setTypeHint(typeHint);
        }
    }

    private void addTypeHintFromEnumProperty(final PropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor.getKind() == ENUM) {
            TypeHint typeHint = getOrCreateTypeHint(propertyDescriptor.getKind());
            typeHint.addEnumValues(propertyDescriptor.getEnumValues());
            setTypeHint(typeHint);
        }
    }

    private void addTypeHintFromPropertyValidationRules(final PropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor instanceof LocalPropertyDescriptor) {
            LocalPropertyDescriptor localPropertyDescriptor = (LocalPropertyDescriptor) propertyDescriptor;
            if (!localPropertyDescriptor.validationRules.isEmpty()) {
                TypeHint typeHint = getOrCreateTypeHint(propertyDescriptor.getKind());
                typeHint.setKind(propertyDescriptor.getKind());
                typeHint.getValidationRules().addAll(localPropertyDescriptor.validationRules);
                setTypeHint(typeHint);
            }
        }
    }

    private void copyTypeHintFromDeployedTypeHint(final PropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor.getTypeHint() != null) {
            setTypeHint(getOrCreateTypeHint(propertyDescriptor.getKind()).copyFrom(propertyDescriptor.getTypeHint()));
        }
    }

    @Override
    public Object get(ConfigurationItem item) {
        return getDeclaringDescriptor().getSyntheticPropertyValue(item, getName());
    }

    @Override
    protected void doSetValue(ConfigurationItem item, Object value) {
        getDeclaringDescriptor().setSyntheticPropertyValue(item, getName(), value);
    }
}
