package com.xebialabs.deployit.booter.local;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.*;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.DeployedSpecific;
import com.xebialabs.deployit.plugin.api.udm.Property;

import static com.xebialabs.deployit.booter.local.utils.Strings.deCamelize;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.BOOLEAN;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.DATE;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.ENUM;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.INTEGER;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.overthere.util.OverthereUtils.checkArgument;
import static java.lang.String.format;

class FieldBasedPropertyDescriptor extends LocalPropertyDescriptor {

    private Field field;

    public FieldBasedPropertyDescriptor(LocalDescriptor descriptor, Field field) {
        this.field = field;
        this.field.setAccessible(true);
        setName(field.getName());
        Property property = field.getAnnotation(Property.class);
        setDeclaringDescriptor(descriptor);
        initMetadata(property);
        initType(property);
        initInspectionMetadata(field);
        reInitializeRequired();
        initValidationMetadata(field);
        setDeployedSpecific(field.isAnnotationPresent(DeployedSpecific.class));
    }

    private void initValidationMetadata(Field field) {
        addDefaultValidationRules();
        for (Annotation annotation : field.getAnnotations()) {
            if (ValidationRuleConverter.isRule(annotation)) {
                this.validationRules.add(ValidationRuleConverter.makeRule(annotation, this));
            }
        }
    }

    private void initInspectionMetadata(Field field) {
        if (field.isAnnotationPresent(InspectionProperty.class)) {
            setInspectionProperty(true);
            setRequiredForInspection(field.getAnnotation(InspectionProperty.class).required());
        }
    }

    private void initMetadata(Property annotation) {
        setCategory(annotation.category());
        setLabel(isBlank(annotation.label()) ? deCamelize(getName()) : annotation.label());
        setDescription(isBlank(annotation.description()) ? getLabel() : annotation.description());
        setPassword(annotation.password());
        setRequired(annotation.required());
        setSize(annotation.size());
        setHidden(annotation.hidden());
        setCandidateValuesFilter(Strings.defaultIfEmpty(annotation.candidateValuesFilter().trim(), null));
        setTransient(isHidden() || annotation.isTransient());
    }

    private void initType(Property annotation) {
        Class<?> type = field.getType();
        if (type == boolean.class) {
            setPrimitiveKind(BOOLEAN);
        } else if (type == int.class) {
            setPrimitiveKind(INTEGER);
        } else if (type == Integer.class) {
            setKind(INTEGER);
        } else if (type == String.class) {
            setKind(STRING);
        } else if (type.isEnum()) {
            setKind(ENUM);
            initEnumValues(field.getType());
        } else if (type == Date.class) {
            setKind(DATE);
        } else if (ConfigurationItem.class.isAssignableFrom(type)) {
            setKind(CI);
            setReferencedType(Type.valueOf(type));
            setAsContainment(annotation.asContainment());
        } else if (Set.class.isAssignableFrom(type)) {
            initSetType(annotation);
        } else if (Map.class.isAssignableFrom(type)) {
            initMapType();
        } else if (List.class.isAssignableFrom(type)) {
            initListType(annotation);
        } else {
            throw new IllegalArgumentException(format("Type of %s not supported as an @Property field, found on %s.%s", type.getName(), field
                    .getDeclaringClass().getName(), getName()));
        }
        registerDefault(Strings.defaultIfEmpty(annotation.defaultValue(), null));
    }

    private void initSetType(Property annotation) {
        Class<?> setType = getGenericType(Set.class, 1, 0);
        if (setType == String.class) {
            setKind(SET_OF_STRING);
        } else if (ConfigurationItem.class.isAssignableFrom(setType)) {
            setKind(SET_OF_CI);
            setReferencedType(Type.valueOf(setType));
            setAsContainment(annotation.asContainment());
        } else {
            throw new IllegalStateException(format("Unsupported Set type encountered for [%s]. Only support String and ConfigurationItem", getName()));
        }
    }

    private void initMapType() {
        checkArgument(getGenericType(Map.class, 2, 0) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), getName());
        checkArgument(getGenericType(Map.class, 2, 1) == String.class, "Property %s.%s of type Map should be Map<String, String>", field.getDeclaringClass().getName(), getName());
        setKind(MAP_STRING_STRING);
    }

    private Class<?> getGenericType(Class<?> collectionClass, int nrExpectedTypes, int indexOfType) {
        java.lang.reflect.Type genericType = field.getGenericType();
        checkArgument(genericType instanceof ParameterizedType, "The field %s.%s is a %s but it isn't a generic type (%s)",
                field.getDeclaringClass().getName(), getName(), collectionClass, genericType);
        java.lang.reflect.Type[] actualTypeArguments = ((ParameterizedType) genericType).getActualTypeArguments();
        checkArgument(actualTypeArguments.length == nrExpectedTypes, "The field %s is a %s.%s but it doesn't have the right generic type (%s)", field.getDeclaringClass()
                .getName(), getName(), collectionClass, actualTypeArguments);
        checkArgument(actualTypeArguments[indexOfType] instanceof Class, "The field %s.%s is a %s but it is not a concrete subclass (%s)", field.getDeclaringClass()
                .getName(), getName(), collectionClass, Arrays.toString(actualTypeArguments));
        return (Class<?>) actualTypeArguments[indexOfType];

    }


    private void initListType(Property annotation) {
        Class<?> listType = getGenericType(List.class, 1, 0);
        if (listType == String.class) {
            setKind(LIST_OF_STRING);
        } else if (ConfigurationItem.class.isAssignableFrom(listType)) {
            setKind(LIST_OF_CI);
            setReferencedType(Type.valueOf(listType));
            setAsContainment(annotation.asContainment());
        } else {
            throw new IllegalStateException(format("Unsupported List type encountered for [%s]. Only support String and ConfigurationItem", getName()));
        }
    }

    protected void reInitializeRequired() {
        super.reInitializeRequired();
        if (field.getType().equals(int.class) && !isRequired()) {
            logger.warn("Optional integer property [{}] backed by a Java int field will be treated as a required property.", this);
            setRequired(true);
        }
    }

    @Override
    public Object get(ConfigurationItem item) {
        try {
            return field.get(item);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Cannot get field " + field, e);
        }
    }

    @Override
    protected void doSetValue(ConfigurationItem item, Object value) {
        try {
            field.set(item, value);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Cannot set field " + field, e);
        }
    }
}
