package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.generator.TypeGenerators;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import nl.javadude.scannit.Scannit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Stream;

import static com.xebialabs.deployit.booter.local.SyntheticHelper.readSyntheticDocument;
import static com.xebialabs.deployit.booter.local.utils.XmlUtils.childByName;
import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;
import static java.io.File.separator;
import static java.util.Collections.list;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;
import static org.apache.commons.lang.StringUtils.EMPTY;

class TypeSystemBootstrapper {

    private final TypeGenerators typeGenerators;
    private static final String PLUGINS = separator + "plugins" + separator;

    TypeSystemBootstrapper(TypeGenerators typeGenerators) {
        this.typeGenerators = typeGenerators;
    }

    void bootstrap(TypeDefinitions typeDefs) {
        logger.info("Initializing type system.");
        scanClasses(typeDefs);
        scanSynthetics(typeDefs);
        createTypeTree(typeDefs);
        registerTypeDefinitions(typeDefs);
        typeGenerators.generateTypeDefinitions(typeDefs);
        createTypeTree(typeDefs);
        registerTypeDefinitions(typeDefs);
    }

    private void createTypeTree(TypeDefinitions typeDefs) {
        typeDefs.getDefinitions().forEach(TypeDefinitions.TypeDefinition::registerTypeTree);
    }

    void verifyTypes(Verifications verifications) {
        DescriptorRegistry.getDescriptors().stream().filter(input -> !input.isVirtual()).forEach((d) -> ((LocalDescriptor) d).verify(verifications));
    }

    private void scanClasses(TypeDefinitions typeDefs) {
        typeDefs.defineType(ConfigurationItem.class);
        Set<Class<? extends ConfigurationItem>> cis = Scannit.getInstance().getSubTypesOf(ConfigurationItem.class);
        for (Class<? extends ConfigurationItem> ci : cis) {
            typeDefs.defineType(ci);
        }
    }

    private void scanSynthetics(TypeDefinitions typeDefs) {
        try {
            List<Element> types = new ArrayList<>();
            List<Element> typeModifications = new ArrayList<>();
            readSynthetics("synthetic.xml", types, typeModifications);
            readSynthetics("synthetic-test.xml", types, typeModifications);

            parseAllTypeDefinitions(types, typeDefs);
            parseAllTypeModifications(typeDefs, typeModifications);
        } catch (IOException ex) {
            throw new RuntimeException("Could not read synthetic type definitions.", ex);
        }
    }

    private void registerTypeDefinitions(TypeDefinitions typeDefs) {
        for (TypeDefinitions.TypeDefinition typeDefinition : typeDefs.getDefinitions()) {
            typeDefinition.register(typeDefs);
        }
    }

    private void parseAllTypeDefinitions(List<Element> types, TypeDefinitions typeDefs) {
        for (Element type : types) {
            typeDefs.defineType(type);
            Iterator<Element> generatedTypes = childByName(type, "generate-deployable"::equals);
            if (generatedTypes.hasNext()) {
                typeDefs.defineGeneratedType(generatedTypes.next(), type);
            }

            findAllGeneratedParameterTypes(typeDefs, type);
        }
    }

    private void findAllGeneratedParameterTypes(final TypeDefinitions typeDefs, final Element type) {
        Iterator<Element> methodDefs = childByName(type, "method"::equals);
        while (methodDefs.hasNext()) {
            Element methodDef = methodDefs.next();
            Iterator<Element> parameters = childByName(methodDef, "parameters"::equals);
            if (parameters.hasNext()) {
                typeDefs.defineGeneratedParameterType(methodDef, type);
            }
        }
    }

    private void parseAllTypeModifications(TypeDefinitions typeDefs, List<Element> typeModifications) {
        for (Element typeModification : typeModifications) {
            typeDefs.modifyType(typeModification);
            findAllGeneratedParameterTypes(typeDefs, typeModification);
        }
    }

    private void readSynthetics(String name, List<Element> typeDefinitionCollector, List<Element> typeModificationsCollector) throws IOException {
        Enumeration<URL> syntheticXMLs = MODULE$.classLoader().getResources(name);

        for (URL syntheticXML : sortedResources(syntheticXMLs)) {
            logger.debug("Scanning synthetic XML: {}", syntheticXML);
            Element docElement = readSyntheticDocument(syntheticXML).getDocumentElement();

            childByName(docElement, "type"::equals).forEachRemaining(typeDefinitionCollector::add);
            childByName(docElement, "type-modification"::equals).forEachRemaining(typeModificationsCollector::add);
        }
    }

    private List<URL> sortedResources(Enumeration<URL> syntheticXMLs) {
        List<URL> files = list(syntheticXMLs);

        return concat(allButPlugins(files), sortedPlugins(files))
                .collect(toList());
    }

    private Stream<URL> allButPlugins(List<URL> files) {
        return files.stream()
                .filter(filterByFolder(PLUGINS).negate());
    }

    private Predicate<URL> filterByFolder(String s) {
        return p -> decode(p).contains(s);
    }

    private String decode(URL url) {
        String decodedString = EMPTY;
        try {
            decodedString = URLDecoder.decode(url.getPath(), "UTF-8");
        } catch (UnsupportedEncodingException ignored) {
        }
        return decodedString;
    }

    private Stream<URL> sortedPlugins(List<URL> files) {
        return files.stream()
            .filter(filterByFolder(PLUGINS))
            .sorted(this::sortByPluginFolderName);
    }

    private int sortByPluginFolderName(URL url1, URL url2) {
        return getPluginFolderName(url1).compareTo(getPluginFolderName(url2));
    }

    private String getPluginFolderName(URL url) {
        String path = decode(url);
        return path.substring(path.lastIndexOf(PLUGINS) + PLUGINS.length());
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeSystemBootstrapper.class);
}
