package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.generator.TypeGenerators;
import com.xebialabs.deployit.plugin.api.reflect.IDescriptorRegistry;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;
import com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument;
import nl.javadude.scannit.Scannit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Stream;


import static com.xebialabs.xlplatform.synthetic.TypeDefinitionDocuments.*;
import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;
import static java.io.File.separator;
import static java.util.Collections.list;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;
import static org.apache.commons.lang.StringUtils.EMPTY;

class TypeSystemBootstrapper {

    private static final String PLUGINS = separator + "plugins" + separator;

    private final Scannit scannit;
    private final TypeDefinitions typeDefinitions;
    private final TypeGenerators typeGenerators;

    TypeSystemBootstrapper(IDescriptorRegistry registry, Scannit scannit) {
        this.scannit = scannit;
        this.typeDefinitions = new TypeDefinitions(registry);
        this.typeGenerators = new TypeGenerators(scannit);
    }

    TypeDefinitions loadTypes() {
        logger.info("Initializing type system.");

        scanClasses();
        scanSynthetics();
        scanZipBasedDefinitions();

        typeDefinitions.registerTypes();
        typeGenerators.generateTypeDefinitions(typeDefinitions);
        typeDefinitions.registerTypes();

        return typeDefinitions;
    }

    //
    // Class-based types
    //

    private void scanClasses() {
        typeDefinitions.defineType(ConfigurationItem.class);
        Set<Class<? extends ConfigurationItem>> cis = scannit.getSubTypesOf(ConfigurationItem.class);
        for (Class<? extends ConfigurationItem> ci : cis) {
            typeDefinitions.defineType(ci);
            TypePluginMappingManager.loadTypePluginMapping(ci, typeDefinitions.getDescriptorRegistry());
        }
    }

    //
    // Synthetic types
    //

    private void scanSynthetics() {
        try {
            List<TypeSpecification> types = new ArrayList<>();
            List<TypeModificationSpecification> typeModifications = new ArrayList<>();
            readSyntheticXml(SYNTHETIC_XML, types, typeModifications);
            readSyntheticXml(SYNTHETIC_TEST_XML, types, typeModifications);

            types.forEach(typeDefinitions::defineType);
            typeModifications.forEach(typeDefinitions::modifyType);
        } catch (IOException ex) {
            throw new RuntimeException("Could not read synthetic type definitions.", ex);
        }
    }

    //
    // Zip Files Type Definitions
    //

    private void scanZipBasedDefinitions() {

        List<TypeSpecification> types = new ArrayList<>();
        try {
            readTypeDefinitionsFromZipForXml(TYPE_DEFINITIONS_XML);
            readTypeDefinitionsFromZipForYaml(TYPE_DEFINITIONS_YAML, types);
        } catch (IOException ex) {
            throw new RuntimeException("Could not read synthetic type definitions.", ex);
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private void readSyntheticXml(String name, List<TypeSpecification> types, List<TypeModificationSpecification> typeModifications) throws IOException {
        Enumeration<URL> syntheticXMLs = MODULE$.classLoader().getResources(name);

        for (URL syntheticXML : sortedResources(syntheticXMLs)) {
            logger.debug("Scanning synthetic XML: {}", syntheticXML);

            SyntheticXmlDocument syntheticXml = SyntheticXmlDocument.read(syntheticXML);

            types.addAll(syntheticXml.getTypes());
            typeModifications.addAll(syntheticXml.getTypeModifications());

            TypePluginMappingManager.loadTypePluginMapping(syntheticXML, syntheticXml);
        }
    }

    private void readTypeDefinitionsFromZipForXml(String name) throws IOException {
        Enumeration<URL> syntheticXMLs = MODULE$.classLoader().getResources(name);

        for (URL syntheticXML : sortedResources(syntheticXMLs)) {
            logger.debug("Scanning Type Defintions in XML: {}", syntheticXML);
            SyntheticXmlDocument syntheticXml = SyntheticXmlDocument.read(syntheticXML);
            TypePluginMappingManager.loadTypePluginMapping(syntheticXML, syntheticXml);
        }
    }

    private void readTypeDefinitionsFromZipForYaml(String name, List<TypeSpecification> types) throws IOException, URISyntaxException {
        Enumeration<URL> syntheticXMLs = MODULE$.classLoader().getResources(name);

        PluginsInZipExtension pluginsInZipExtension = new PluginsInZipExtension();

        for (URL syntheticXML : sortedResources(syntheticXMLs)) {
            logger.debug("Scanning Type Defintions in YAML: {}", syntheticXML);
            String filePath = syntheticXML.getPath().replace("!/"+TYPE_DEFINITIONS_YAML, "");
            if(filePath.endsWith(".zip")){
                File zipFile = new File(URI.create(filePath));
                if(zipFile.exists()){
                    var result = pluginsInZipExtension.extractFrom(zipFile);
                    result.get().get().foreach(types::add);
                    TypePluginMappingManager.loadTypePluginMapping(syntheticXML, result.get().get());
                }
            }
        }
    }

    private List<URL> sortedResources(Enumeration<URL> syntheticXMLs) {
        List<URL> files = list(syntheticXMLs);

        return concat(allButPlugins(files), sortedPlugins(files))
                .collect(toList());
    }

    private Stream<URL> allButPlugins(List<URL> files) {
        return files.stream()
                .filter(filterByFolder(PLUGINS).negate());
    }

    private Predicate<URL> filterByFolder(String s) {
        return p -> decode(p).contains(s);
    }

    private String decode(URL url) {
        String decodedString = EMPTY;
        try {
            decodedString = URLDecoder.decode(url.getPath(), "UTF-8");
        } catch (UnsupportedEncodingException ignored) {
        }
        return decodedString;
    }

    private Stream<URL> sortedPlugins(List<URL> files) {
        return files.stream()
                .filter(filterByFolder(PLUGINS))
                .sorted(this::sortByPluginFolderName);
    }

    private int sortByPluginFolderName(URL url1, URL url2) {
        return getPluginFolderName(url1).compareTo(getPluginFolderName(url2));
    }

    private String getPluginFolderName(URL url) {
        String path = decode(url);
        return path.substring(path.lastIndexOf(PLUGINS) + PLUGINS.length());
    }

    private static final Logger logger = LoggerFactory.getLogger(TypeSystemBootstrapper.class);
}
