package com.xebialabs.xlplatform.synthetic.xml;

import java.io.IOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import com.xebialabs.xlplatform.synthetic.TypeModificationSpecification;
import com.xebialabs.xlplatform.synthetic.TypeSpecification;

import static com.xebialabs.xlplatform.utils.ClassLoaderUtils$.MODULE$;
import static javax.xml.XMLConstants.W3C_XML_SCHEMA_NS_URI;

public class SyntheticXmlDocument implements XmlOperations {

    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY = createDocumentBuilderFactory();
    private static final Logger logger = LoggerFactory.getLogger(SyntheticXmlDocument.class);

    private final List<TypeSpecification> types = new ArrayList<>();
    private final List<TypeModificationSpecification> typeModifications = new ArrayList<>();

    private SyntheticXmlDocument(Element docElement) {
        childByName(docElement, "type"::equals)
                .forEachRemaining(element -> types.add(new XmlTypeSpecification(element)));
        childByName(docElement, "type-modification"::equals)
                .forEachRemaining(element -> typeModifications.add(new XmlTypeModificationSpecification(element)));
    }

    public static SyntheticXmlDocument read(URL syntheticXML) throws IOException {
        String syntheticString = IOUtils.toString(syntheticXML, StandardCharsets.UTF_8);
        Element docElement = readSyntheticDocument(syntheticString).getDocumentElement();
        return new SyntheticXmlDocument(docElement);
    }

    public static SyntheticXmlDocument read(String syntheticXML) throws IOException {
        Element docElement = readSyntheticDocument(syntheticXML).getDocumentElement();
        return new SyntheticXmlDocument(docElement);
    }

    public List<TypeSpecification> getTypes() {
        return types;
    }

    public List<TypeModificationSpecification> getTypeModifications() {
        return typeModifications;
    }

    //
    // Parse XML
    //

    private static DocumentBuilderFactory createDocumentBuilderFactory() {
        try {
            SchemaFactory schemaFactory = SchemaFactory.newInstance(W3C_XML_SCHEMA_NS_URI);
            Schema syntheticSchema = schemaFactory.newSchema(MODULE$.classLoader().getResource("synthetic.xsd"));
            DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
            documentBuilderFactory.setNamespaceAware(true);
            documentBuilderFactory.setXIncludeAware(false);
            documentBuilderFactory.setExpandEntityReferences(false);
            documentBuilderFactory.setSchema(syntheticSchema);

            setFeatureQuietly(documentBuilderFactory, XMLConstants.FEATURE_SECURE_PROCESSING, true);
            setFeatureQuietly(documentBuilderFactory, "http://apache.org/xml/features/disallow-doctype-decl", true);
            setFeatureQuietly(documentBuilderFactory, "http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
            setFeatureQuietly(documentBuilderFactory, "http://xml.org/sax/features/external-general-entities", false);
            setFeatureQuietly(documentBuilderFactory, "http://xml.org/sax/features/external-parameter-entities", false);
            return documentBuilderFactory;
        } catch (SAXException exc) {
            throw new IllegalStateException("Cannot read schema synthetic.xsd", exc);
        }
    }

    private static void setFeatureQuietly(DocumentBuilderFactory factory, String name, boolean value) {
        try {
            factory.setFeature(name, value);
        } catch (Exception ex) {
            logger.warn(String.format("Cannot set feature [%s] for [%s]", name, factory.getClass().getName()), ex);
        }
    }

    private static Document readSyntheticDocument(final String syntheticXML) throws IOException {
        try {
            final boolean[] validationErrorsFound = new boolean[1];
            DocumentBuilder builder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder();
            builder.setErrorHandler(new ErrorHandler() {
                @Override
                public void warning(SAXParseException exc) {
                    logger.warn("Warning while parsing " + syntheticXML, exc);
                }

                @Override
                public void error(SAXParseException exc) {
                    logger.error("Error while parsing " + syntheticXML, exc);
                    validationErrorsFound[0] = true;
                }

                @Override
                public void fatalError(SAXParseException exc) {
                    logger.error("Fatal error while parsing " + syntheticXML, exc);
                    validationErrorsFound[0] = true;
                }
            });
            Document doc = builder.parse(IOUtils.toInputStream(syntheticXML, StandardCharsets.UTF_8));
            if (validationErrorsFound[0]) {
                throw new IllegalArgumentException("One or more errors were found while parsing " + syntheticXML);
            }

            return doc;
        } catch (RuntimeException | ParserConfigurationException | SAXException exc) {
            throw new IllegalStateException("Cannot read synthetic configuration " + syntheticXML, exc);
        }
    }

}
